/******************************************************************************/
/*                                                                            */
/*                             WorldSkills France                             */
/*                       48th edition - Marseille 2025                        */
/*                       Electronique (16) - SyncOrSink                       */
/*                                                                            */
/******************************************************************************/
/* @file CONF/CONF_SystemStates.c                                             */
/* @authors WorldSkills France "Electronique" skill team                      */
/* @version 1.0                                                               */
/* @date 2025-09-06                                                           */
/*                                                                            */
/* @brief This file contains the functions to select system global state.     */
/******************************************************************************/

/* *************************** STANDARD INCLUDES **************************** */

/* **************************** CUSTOM INCLUDES ***************************** */

#include "CONF_SystemStates.h"

/* ******************************** DEFINES ********************************* */

/* *************************** TYPES DECLARATION **************************** */

/* ******************************* CONSTANTS ******************************** */

/* **************************** GLOBAL VARIABLES **************************** */

/* **************************** STATIC VARIABLES **************************** */

/* ********************** STATIC FUNCTIONS DECLARATION ********************** */

/* ************************* FUNCTIONS DECLARATION ************************** */

/* ********************** STATIC FUNCTIONS DEFINITION *********************** */

/* ************************** FUNCTIONS DEFINITION ************************** */

/******************************************************************************/
/* @function CONF_SystemState_Select                                          */
/*                                                                            */
/* @brief Selects the new system state based on previous state.               */
/* @param [in] previousState Previous system state                            */
/* @param [in] btnDispStatus DISP button activation status                    */
/* @retval New system state                                                   */
/* @req SYS_REQ-0306-001 : Changement d’état du système                       */
/******************************************************************************/
tCONF_SystemStates CONF_SystemState_Select
(
    const tCONF_SystemStates previousState,
    const bool btnDispStatus
)
{
    tCONF_SystemStates newState;

    /* When pressing the BTN_DISP button, the system state shall change to    */
    /* the next state. Once all states have been cycled through, the system   */
    /* shall return to state Home.                                            */
    if(btnDispStatus)
    {
        newState = previousState + 1;
        if(newState >= E_NB_STATES)
        {
            newState = E_STATE_HOME;
        }
    }
    else
    {
        newState = previousState;
    }
    return newState;
}

/******************************************************************************/
/* @function CONF_UpdateContextWithMODE                                       */
/*                                                                            */
/* @brief Updates system context based on current system state and MODE       */
/*        button activation status.                                           */
/* @param [in] currentState Current system state                              */
/* @param [in] btnModeStatus MODE button activation status                    */
/* @param [in|out] calendar Calendar used by the system                       */
/* @param [in|out] rfCarrierActive RF carrier activation status               */
/* @req SYS_REQ-0307-001 : Opérations en état STATE_DATE                      */
/* @req SYS_REQ-0311-001 : Opérations en état STATE_CAR                       */
/******************************************************************************/
void CONF_UpdateContextWithMODE
(
    const tCONF_SystemStates currentState,
    const bool btnModeStatus,
    tTIME_Calendar * const calendar,
    bool * const rfCarrierActive
)
{
    if(btnModeStatus)
    {
        switch(currentState)
        {
            case E_STATE_DATE:
                /* If the MODE button is pressed and the current state set to */
                /* "DATE", changes the calendar.                              */
                *calendar += 1;
                if((*calendar <= E_CALENDAR_UNKNOWN) || (*calendar >= E_NB_CALENDAR))
                {
                    *calendar = E_CALENDAR_JULIAN;
                }
                break;

            case E_STATE_CAR:
                /* If the MODE button is pressed and the current state set to */
                /* "RF carrier", the RF carrier activation status is inverted.*/
                *rfCarrierActive = !(*rfCarrierActive);
                break;

            default:
                break;
        }
    }
}

/******************************************************************************/
/*                                                                            */
/*                             WorldSkills France                             */
/*                       48th edition - Marseille 2025                        */
/*                       Electronique (16) - SyncOrSink                       */
/*                                                                            */
/******************************************************************************/
