/******************************************************************************/
/*                                                                            */
/*                             WorldSkills France                             */
/*                       48th edition - Marseille 2025                        */
/*                       Electronique (16) - SyncOrSink                       */
/*                                                                            */
/******************************************************************************/
/* @file SUP/SUP_ApplyCmds.c                                                  */
/* @authors WorldSkills France "Electronique" skill team                      */
/* @version 1.0                                                               */
/* @date 2025-09-14                                                           */
/*                                                                            */
/* @brief This file contains functions to apply administration commands.      */
/******************************************************************************/

/* *************************** STANDARD INCLUDES **************************** */

#include <stdlib.h>
#include <string.h>

/* **************************** CUSTOM INCLUDES ***************************** */

#include "SUP_ApplyCmds.h"

/* ******************************** DEFINES ********************************* */

/* *************************** TYPES DECLARATION **************************** */

/* ******************************* CONSTANTS ******************************** */

/* **************************** GLOBAL VARIABLES **************************** */

/* **************************** STATIC VARIABLES **************************** */

/* ********************** STATIC FUNCTIONS DECLARATION ********************** */

/* ************************* FUNCTIONS DECLARATION ************************** */

/* ********************** STATIC FUNCTIONS DEFINITION *********************** */

/* ************************** FUNCTIONS DEFINITION ************************** */

/******************************************************************************/
/* @function SUP_ProcessAdminCommands                                         */
/*                                                                            */
/* @brief Processes the administration commands.                              */
/* @param [in] nbCmdsReceived Number of commands received from RS_ADMIN line  */
/* @param [in] cmds Commands received from RS_ADMIN line                      */
/* @param [out] pingIds PING commands identifiers                             */
/* @param [out] pNewCalendar New calendar from CALD command                   */
/* @param [out] pNewTimestamp New timestamp from TMSP command                 */
/* @req SYS_REQ-0205-001 : Traitement des commandes d’administration – Commande PING */
/* @req SYS_REQ-0206-001 : Traitement des commandes d’administration – Commande CALD */
/* @req SYS_REQ-0207-001 : Traitement des commandes d’administration – Commande TMSP */
/******************************************************************************/
void SUP_ProcessAdminCommands
(
    const unsigned int nbCmdsReceived,
    const tRsAdmin_Cmd cmds[C_RSADMININ_MAX_CMDS],
    tPingIds pingIds,
    tTIME_Calendar * pNewCalendar,
    tTIME_Timestamp * pNewTimestamp
)
{
    unsigned int c;
    int pingIdx;

    /* ====================================================================== */
    /* Outputs init                                                           */
    /* ====================================================================== */

    memset(pingIds, 0, sizeof(tPingIds));
    *pNewCalendar  = E_NB_CALENDAR;
    *pNewTimestamp = C_UNDEFINED_TIMESTAMP;

    /* ====================================================================== */
    /* Commands processing                                                    */
    /* ====================================================================== */

    pingIdx = 0;
    for(c = 0 ; c < nbCmdsReceived ; ++c)
    {
        if(cmds[c].used) /* Note: command is used only if its content is valid */
        {
            switch(cmds[c].code)
            {
                case E_COM_RSADMIN_CODE_PING:
                    /* ------------------------------------------------------ */
                    /* If one or more valid PING commands have been received, */
                    /* PING_IDS shall contain the list of IDs of received     */
                    /* PING commands (ordered by reception order). If no PING */
                    /* command has been received, PING_IDS shall be an empty  */
                    /* list.                                                  */
                    /* ------------------------------------------------------ */
                    pingIds[pingIdx][0] = cmds[c].id[0];
                    pingIds[pingIdx][1] = cmds[c].id[1];
                    pingIdx += 1;
                    break;

                case E_COM_RSADMIN_CODE_CALD:
                    /* ------------------------------------------------------ */
                    /* If a valid CALD command has been received, command     */
                    /* data shall be decoded and NEW_CALENDAR output set with */
                    /* obtained calendar. Otherwise, NEW_CALENDAR shall be    */
                    /* set with default value.                                */
                    /* ------------------------------------------------------ */
                    switch(cmds[c].data[0])
                    {
                        case '1': *pNewCalendar = E_CALENDAR_JULIAN   ; break;
                        case '2': *pNewCalendar = E_CALENDAR_GREGORIAN; break;
                        case '3': *pNewCalendar = E_CALENDAR_UTC      ; break;
                        default : *pNewCalendar = E_CALENDAR_UNKNOWN  ; break;
                    }
                    break;

                case E_COM_RSADMIN_CODE_TMSP:
                    /* ------------------------------------------------------ */
                    /* If a valid TMSP command has been received, command     */
                    /* data shall be decoded and NEW_TIMESTAMP output set     */
                    /* with obtained calendar. Otherwise, NEW_TIMESTAMP shall */
                    /* be set with default value.                             */
                    /* ------------------------------------------------------ */
                    *pNewTimestamp = atoll(cmds[c].data);
                    break;

                default:
                    break;
            }
        }
    }
}

/******************************************************************************/
/*                                                                            */
/*                             WorldSkills France                             */
/*                       48th edition - Marseille 2025                        */
/*                       Electronique (16) - SyncOrSink                       */
/*                                                                            */
/******************************************************************************/
