/******************************************************************************/
/*                                                                            */
/*                             WorldSkills France                             */
/*                       48th edition - Marseille 2025                        */
/*                       Electronique (16) - SyncOrSink                       */
/*                                                                            */
/******************************************************************************/
/* @file SUP/SUP_PushButtons.c                                                */
/* @authors WorldSkills France "Electronique" skill team                      */
/* @version 1.0                                                               */
/* @date 2025-09-06                                                           */
/*                                                                            */
/* @brief This file contains the context and functions to handle push buttons.*/
/******************************************************************************/

/* *************************** STANDARD INCLUDES **************************** */

/* **************************** CUSTOM INCLUDES ***************************** */

#include "SUP_PushButtons.h"

/* ******************************** DEFINES ********************************* */

/* *************************** TYPES DECLARATION **************************** */

/* ******************************* CONSTANTS ******************************** */

/* **************************** GLOBAL VARIABLES **************************** */

tPushButton BtnDisp_handler;
tPushButton BtnMode_handler;

/* **************************** STATIC VARIABLES **************************** */

/* ********************** STATIC FUNCTIONS DECLARATION ********************** */

/* ************************* FUNCTIONS DECLARATION ************************** */

/* ********************** STATIC FUNCTIONS DEFINITION *********************** */

/* ************************** FUNCTIONS DEFINITION ************************** */

/******************************************************************************/
/* @function SUP_PushButtons_ReadCurrentStates                                */
/*                                                                            */
/* @brief Read MODE and DISP push buttons states.                             */
/* @param [out] V_BTN_MODE MODE push button state (TRUE = activated)          */
/* @param [out] V_BTN_DISP DISP push button state (TRUE = activated)          */
/* @pre Buttons shall have been configured by SUP_PushButtons_Init function.  */
/* @req SYS_REQ-0208-002 : Bouton utilisateur "MODE" > Lecture de l'état      */
/* @req SYS_REQ-0209-002 : Bouton utilisateur "DISP" > Lecture de l'état      */
/******************************************************************************/
void SUP_PushButtons_ReadCurrentStates
(
    bool * const V_BTN_MODE,
    bool * const V_BTN_DISP
)
{
    /* ---- Temporary disabling TIM22 interrupts to copy button states ------ */
    HAL_NVIC_DisableIRQ(TIM22_IRQn);

    /* ---- Setting button status output ------------------------------------ */
    *V_BTN_MODE = (BtnMode_handler.lstate == PBLS_ACTIVE);
    *V_BTN_DISP = (BtnDisp_handler.lstate == PBLS_ACTIVE);

    /* ---- Acknowledgement of rising edge detection ------------------------ */
    BtnMode_handler.edge_ack = true;
    BtnDisp_handler.edge_ack = true;

    /* ---- Re-enabling TIM22 interrupts ------------------------------------ */
    HAL_NVIC_EnableIRQ(TIM22_IRQn);
}

/******************************************************************************/
/* @function SUP_PushButtons_Init                                             */
/*                                                                            */
/* @brief Initializes the DISP and MODE push buttons.                         */
/* @param [in] GPIO_Port_Btn_Mode MODE button GPIO port                       */
/* @param [in] GPIO_Pin_Btn_Mode  MODE button GPIO pin                        */
/* @param [in] GPIO_Port_Btn_Disp DISP button GPIO port                       */
/* @param [in] GPIO_Pin_Btn_Disp  DISP button GPIO pin                        */
/* @req SYS_REQ-0208-002 : Bouton utilisateur "MODE" > Configuration          */
/* @req SYS_REQ-0209-002 : Bouton utilisateur "DISP" > Configuration          */
/******************************************************************************/
void SUP_PushButtons_Init
(
    GPIO_TypeDef *const GPIO_Port_Btn_Mode,
    const uint16_t GPIO_Pin_Btn_Mode,
    GPIO_TypeDef *const GPIO_Port_Btn_Disp,
    const uint16_t GPIO_Pin_Btn_Disp
)
{
    /* ====================================================================== */
    /* Button state reading frequency                                         */
    /* ====================================================================== */
    /* Button states are read by TIM22. TIM22 interrupt frequency is set by   */
    /* MX_TIM22_Init and button states are read by TIM22_IRQHandler function. */
    /* TIM22 config is set as follows:                                        */
    /* - Fclk = 32 MHz                                                        */
    /* - PSC  = 7                                                             */
    /* - ARR  = 39999                                                         */
    /* Finterrupt = Fclk / ((PSC+1)(ARR+1)) = 32.10^6/(8*40000) = 100 Hz      */
    /*                                                          => T = 10 ms  */
    /* ---------------------------------------------------------------------- */

    /* ====================================================================== */
    /* Button state reading mode : rising edge                                */
    /* ====================================================================== */
    PushButton_init(&BtnDisp_handler, GPIO_Port_Btn_Disp, GPIO_Pin_Btn_Disp, PBM_RISING_EDGE);
    PushButton_init(&BtnMode_handler, GPIO_Port_Btn_Mode, GPIO_Pin_Btn_Mode, PBM_RISING_EDGE);
}

/******************************************************************************/
/*                                                                            */
/*                             WorldSkills France                             */
/*                       48th edition - Marseille 2025                        */
/*                       Electronique (16) - SyncOrSink                       */
/*                                                                            */
/******************************************************************************/
