/******************************************************************************/
/*                                                                            */
/*                             WorldSkills France                             */
/*                       48th edition - Marseille 2025                        */
/*                       Electronique (16) - SyncOrSink                       */
/*                                                                            */
/******************************************************************************/
/* @file SUP/SUP_Reception.c                                                  */
/* @authors WorldSkills France "Electronique" skill team                      */
/* @version 1.0                                                               */
/* @date 2025-09-12                                                           */
/*                                                                            */
/* @brief This file contains functions to receive RS_ADMIN commands.          */
/******************************************************************************/

/* *************************** STANDARD INCLUDES **************************** */

#include <string.h>

/* **************************** CUSTOM INCLUDES ***************************** */

#include "SUP_Reception.h"

/* ******************************** DEFINES ********************************* */

/* *************************** TYPES DECLARATION **************************** */

/* ******************************* CONSTANTS ******************************** */

/* **************************** GLOBAL VARIABLES **************************** */

/* **************************** STATIC VARIABLES **************************** */

static UART_HandleTypeDef * __RSADMIN_RECEPTION_UART_HANDLER;
static tRsAdminBuffer __RSADMIN_RECEPTION_DMA_BUFFER; /* Volatile buffer */

/* ********************** STATIC FUNCTIONS DECLARATION ********************** */

/* ************************* FUNCTIONS DECLARATION ************************** */

/* ********************** STATIC FUNCTIONS DEFINITION *********************** */

/* ************************** FUNCTIONS DEFINITION ************************** */

/******************************************************************************/
/* @function SUP_Reception_Init                                               */
/*                                                                            */
/* @brief Initializes the transmission interface for remote equipments.       */
/* @param [in] UartHandler UART serial reception line handler                 */
/* @req SYS_REQ-0201-002 : Interface RS administration > Configuration        */
/******************************************************************************/
void SUP_Reception_Init(UART_HandleTypeDef * const UartHandler)
{
    /* ====================================================================== */
    /* The system shall provide an unidirectional UART serial reception       */
    /* interface to receive administration commands. This serial interface    */
    /* shall be configured as follows:                                        */
    /* - Baude rate : 9600 bauds                                              */
    /* - 8 data bits                                                          */
    /* - Odd parity bit                                                       */
    /* - 1 stop bit                                                           */
    /* ====================================================================== */
    /* The serial line configuration is generated automagically in the        */
    /* MX_USART1_UART_Init function.                                          */
    /* ---------------------------------------------------------------------- */
    __RSADMIN_RECEPTION_UART_HANDLER = UartHandler;

    /* ---------------------------------------------------------------------- */
    /* There are two methods to read data received on RS_ADMIN_IN:            */
    /* - Polling method, which can only read data when called and blocks the  */
    /*   execution until data is received or its timeout expired.             */
    /* - Continous reading with DMA copy, which continuously reads the UART   */
    /*   line and store the data received in an array.                        */
    /*                                                                        */
    /* The DMA method is more complex than the polling method, but have many  */
    /* benefits:                                                              */
    /* - The values are always available and there is no need to wait for the */
    /*   polling function to wait and read them (the reading is done in       */
    /*   background, while polling method starts to read the values only when */
    /*   asked to).                                                           */
    /* - No data is loss, as the DMA continuously reads the values thanks to  */
    /*   hardware features.                                                   */
    /* - The continuous reading is a hardware feature, and DMA only copies    */
    /*   values to RAM when they are ready, which means no CPU usage at all!  */
    /* ---------------------------------------------------------------------- */
    memset(__RSADMIN_RECEPTION_DMA_BUFFER, 0, sizeof(tRsAdminBuffer));
    HAL_UART_Receive_DMA(__RSADMIN_RECEPTION_UART_HANDLER, (tSTM32_CHAR_ARRAY)__RSADMIN_RECEPTION_DMA_BUFFER, sizeof(tRsAdminBuffer));
}

/******************************************************************************/
/* @function SUP_Reception_ReceiveData                                        */
/*                                                                            */
/* @brief Receives data from RS_ADMIN_IN RS line.                             */
/* @param [in] buffer Content received on RS_ADMIN_IN reception line          */
/* @req SYS_REQ-0201-002 : Interface RS administration > Reception            */
/******************************************************************************/
void SUP_Reception_ReceiveData(tRsAdminBuffer buffer)
{
    if(__RSADMIN_RECEPTION_DMA_BUFFER[0] != '\0')
    {
        /* ------------------------------------------------------------------ */
        /* If the reception buffer is not empty, stops the DMA from writing   */
        /* in __COM_RECEPTION_DMA_BUFFER (to avoid concurrent access), copies */
        /* the data received, then flushes the buffer and restarts the DMA at */
        /* the beginning of the array.                                        */
        /*                                                                    */
        /* Note: this technic may lead to data loss. If data are transmitted  */
        /* over RS_ADMIN_IN during recopy, the data are lost as the DMA is    */
        /* stopped. However, in our application, RS_ADMIN_IN is expected to   */
        /* be written by a human user, so we consider that the next data will */
        /* arrive long after the DMA is restarted. There is no need for more  */
        /* robustness in this application.                                    */
        /* ------------------------------------------------------------------ */
        HAL_UART_DMAStop(__RSADMIN_RECEPTION_UART_HANDLER);
        memcpy(buffer, __RSADMIN_RECEPTION_DMA_BUFFER, sizeof(tRsAdminBuffer));
        memset(__RSADMIN_RECEPTION_DMA_BUFFER, 0, sizeof(tRsAdminBuffer));
        HAL_UART_Receive_DMA(__RSADMIN_RECEPTION_UART_HANDLER, (tSTM32_CHAR_ARRAY)__RSADMIN_RECEPTION_DMA_BUFFER, sizeof(tRsAdminBuffer));
    }
    else if(buffer[0] != '\0')
    {
        /* ------------------------------------------------------------------ */
        /* If no data has been received but the output buffer is not empty,   */
        /* flushes its content.                                               */
        /* ------------------------------------------------------------------ */
        memset(buffer, 0, sizeof(tRsAdminBuffer));
    }
}

/******************************************************************************/
/*                                                                            */
/*                             WorldSkills France                             */
/*                       48th edition - Marseille 2025                        */
/*                       Electronique (16) - SyncOrSink                       */
/*                                                                            */
/******************************************************************************/
