/******************************************************************************/
/*                                                                            */
/*                             WorldSkills France                             */
/*                       48th edition - Marseille 2025                        */
/*                       Electronique (16) - SyncOrSink                       */
/*                                                                            */
/******************************************************************************/
/* @file SUP/SUP_RsAdmin.c                                                    */
/* @authors WorldSkills France "Electronique" skill team                      */
/* @version 1.0                                                               */
/* @date 2025-09-12                                                           */
/*                                                                            */
/* @brief This file contains functions to decode RS_ADMIN_IN messages.        */
/******************************************************************************/

/* *************************** STANDARD INCLUDES **************************** */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* **************************** CUSTOM INCLUDES ***************************** */

#include "SUP_RsAdmin.h"

/* ******************************** DEFINES ********************************* */

#define __C_RSADMIN_MESSAGE_SEPARATOR_START ('$')
#define __C_RSADMIN_MESSAGE_SEPARATOR_BLOCK (',')
#define __C_RSADMIN_MESSAGE_SEPARATOR_END   (';')

#define __C_RSADMIN_NB_FIELDS_PING (2)
#define __C_RSADMIN_NB_FIELDS_CALD (3)
#define __C_RSADMIN_NB_FIELDS_TMSP (3)

/* *************************** TYPES DECLARATION **************************** */

/* ******************************* CONSTANTS ******************************** */

#define __C_RSADMIN_FIELD_IDT_POS (0)
#define __C_RSADMIN_FIELD_NUM_POS (1)

#define __C_RSADMIN_FIELD_CAL_POS (2)
#define __C_RSADMIN_FIELD_TSP_POS (2)

/* **************************** GLOBAL VARIABLES **************************** */

/* **************************** STATIC VARIABLES **************************** */

/* ********************** STATIC FUNCTIONS DECLARATION ********************** */

/* ************************* FUNCTIONS DECLARATION ************************** */

/* ********************** STATIC FUNCTIONS DEFINITION *********************** */

/******************************************************************************/
/* @function __COM_Messages_VerifyValidity                                    */
/*                                                                            */
/* @brief Verifies the validity of a RS_ADMIN command.                        */
/* @param [in] cmdstr Blocks of strings representing the received command     */
/* @param [in] expectedNbFields Expected number of fields in the command      */
/* @param [out] cmdId Command unique identifier                               */
/* @pre The command code (IDT field) must have firstly been verified.         */
/* @req SYS_REQ-0202-001 : Réception des commandes d’administration           */
/******************************************************************************/
static tRsAdmin_Validity __COM_Messages_VerifyValidity
(
    const char cmdstr[C_RSADMININ_MAX_BLOCK_NB][C_RSADMININ_MAX_BLOCK_LENGTH],
    const unsigned int expectedNbFields,
    tCmdId cmdId
)
{
    tRsAdmin_Validity validity = E_COM_CMD_OK;

    /* ====================================================================== */
    /* Verifying command consistency                                          */
    /* ====================================================================== */
    /* The command is considerer consistent if the following conditions are   */
    /* met:                                                                   */
    /* - It contains the right number of fields (number of fields + 1 is      */
    /*   empty, and number of fields is not)                                  */
    /* - The first block "IDT" is the message code (not verified as it is a   */
    /*   precondition to the function call)                                   */
    /* - The second block is a one or two (not more) characters identifier    */
    /* ====================================================================== */
    if
    (
        cmdstr[expectedNbFields][0] != '\0'
        || cmdstr[__C_RSADMIN_FIELD_NUM_POS][0] == '\0'
        || cmdstr[__C_RSADMIN_FIELD_NUM_POS][2] != '\0'
    )
    {
        validity = E_COM_VALIDITY_FORMAT_ERROR;
    }

    /* ====================================================================== */
    /* Extracting command ID                                                  */
    /* ====================================================================== */
    cmdId[0] = cmdstr[__C_RSADMIN_FIELD_NUM_POS][0];
    cmdId[1] = cmdstr[__C_RSADMIN_FIELD_NUM_POS][1];

    return validity;
}

/******************************************************************************/
/* @function __COM_Messages_Init_RSADMIN_Cmd                                  */
/*                                                                            */
/* @brief Initializes a RS_ADMIN command                                      */
/* @param [out] pCmd Initialized RS_ADMIN command                             */
/* @req None                                                                  */
/******************************************************************************/
static void __COM_Messages_Init_RSADMIN_Cmd(tRsAdmin_Cmd * const pCmd)
{
    pCmd->validity = E_COM_VALIDITY_UNDEFINED;
    pCmd->used = false;
    pCmd->code = E_COM_RSADMIN_CODE_UNDEFINED;
    memset(pCmd->id, 0, sizeof(tCmdId));
    memset(pCmd->data, 0, C_RSADMININ_LENGTH_DATA * sizeof(char));
}

/******************************************************************************/
/* @function __COM_Messages_Decode_RSADMIN_PING                               */
/*                                                                            */
/* @brief Decodes a "PING" command.                                           */
/* @param [in] cmdstr Blocks of strings representing the received command     */
/* @param [out] pCmd  Command built from the string command                   */
/* @pre The command must be a "PING" command.                                 */
/* @req SYS_REQ-0202-001 : Réception des commandes d’administration > PING    */
/* @req SYS_REQ-0203-001 : Décodage d’une commande d’administration > PING    */
/******************************************************************************/
static void __COM_Messages_Decode_RSADMIN_PING
(
    const char cmdstr[C_RSADMININ_MAX_BLOCK_NB][C_RSADMININ_MAX_BLOCK_LENGTH],
    tRsAdmin_Cmd * const pCmd
)
{
    pCmd->code = E_COM_RSADMIN_CODE_PING;
    pCmd->validity = __COM_Messages_VerifyValidity(cmdstr, __C_RSADMIN_NB_FIELDS_PING, pCmd->id);
    /* No data to retrieve as no data is expected for PING command. */
}

/******************************************************************************/
/* @function __COM_Messages_Decode_RSADMIN_CALD                               */
/*                                                                            */
/* @brief Decodes a "CALD" command.                                           */
/* @param [in] cmdstr Blocks of strings representing the received command     */
/* @param [out] pCmd  Command built from the string command                   */
/* @pre The command must be a "CALD" command.                                 */
/* @req SYS_REQ-0202-001 : Réception des commandes d’administration > CALD    */
/* @req SYS_REQ-0203-001 : Décodage d’une commande d’administration > CALD    */
/******************************************************************************/
static void __COM_Messages_Decode_RSADMIN_CALD
(
    const char cmdstr[C_RSADMININ_MAX_BLOCK_NB][C_RSADMININ_MAX_BLOCK_LENGTH],
    tRsAdmin_Cmd * const pCmd
)
{
    pCmd->code = E_COM_RSADMIN_CODE_CALD;
    pCmd->validity = __COM_Messages_VerifyValidity(cmdstr, __C_RSADMIN_NB_FIELDS_CALD, pCmd->id);

    /* ====================================================================== */
    /* Extracting command data                                                */
    /* ====================================================================== */
    memcpy(pCmd->data, cmdstr[__C_RSADMIN_FIELD_CAL_POS], C_RSADMININ_LENGTH_DATA * sizeof(char));
    pCmd->data[C_RSADMININ_LENGTH_DATA - 1] = '\0';

    /* ====================================================================== */
    /* Verifying data ranges                                                  */
    /* ====================================================================== */
    /* "CAL" field is a one-character digit between 1 and 3.                  */
    /* - If data field contains more than one character, the field is out of  */
    /*   range.                                                               */
    /* - If the value is not 1, 2 or 3, the field is out of range.            */
    /* ---------------------------------------------------------------------- */
    if
    (
        pCmd->data[1] != '\0'
        || (pCmd->data[0] != '1' && pCmd->data[0] != '2' && pCmd->data[0] != '3')
    )
    { pCmd->validity = E_COM_VALIDITY_OUT_OF_RANGE; }
}

/******************************************************************************/
/* @function __COM_Messages_Decode_RSADMIN_TMSP                               */
/*                                                                            */
/* @brief Decodes a "TMSP" command.                                           */
/* @param [in] cmdstr Blocks of strings representing the received command     */
/* @param [out] pCmd  Command built from the string command                   */
/* @pre The command must be a "TMSP" command.                                 */
/* @req SYS_REQ-0202-001 : Réception des commandes d’administration > CALD    */
/* @req SYS_REQ-0203-001 : Décodage d’une commande d’administration > CALD    */
/******************************************************************************/
static void __COM_Messages_Decode_RSADMIN_TMSP
(
    const char cmdstr[C_RSADMININ_MAX_BLOCK_NB][C_RSADMININ_MAX_BLOCK_LENGTH],
    tRsAdmin_Cmd * const pCmd
)
{
    unsigned int c;
    bool isANumber;

    pCmd->code = E_COM_RSADMIN_CODE_TMSP;
    pCmd->validity = __COM_Messages_VerifyValidity(cmdstr, __C_RSADMIN_NB_FIELDS_TMSP, pCmd->id);

    /* ====================================================================== */
    /* Extracting command data                                                */
    /* ====================================================================== */
    memcpy(pCmd->data, cmdstr[__C_RSADMIN_FIELD_TSP_POS], C_RSADMININ_LENGTH_DATA * sizeof(char));
    pCmd->data[C_RSADMININ_LENGTH_DATA - 1] = '\0';

    /* ====================================================================== */
    /* Verifying data ranges                                                  */
    /* ====================================================================== */
    /* "TSP" field is expected to be a 1 to 10 digits positive integer        */
    /* ---------------------------------------------------------------------- */
    if(pCmd->validity == E_COM_CMD_OK)
    {
        isANumber = true;
        c = 0;
        while(pCmd->data[c] != '\0' && c < C_RSADMININ_LENGTH_DATA)
        {
            if(pCmd->data[c] < '0' || pCmd->data[c] > '9')
            {
                isANumber = false;
                break;
            }
            c += 1;
        }

        if(!isANumber) { pCmd->validity = E_COM_VALIDITY_FORMAT_ERROR; }
        else if(pCmd->data[C_RSADMININ_LENGTH_DATA - 1] != '\0') { pCmd->validity = E_COM_VALIDITY_OUT_OF_RANGE; }
    }
}

/******************************************************************************/
/* @function __COM_Messages_RetrieveCommandFromBlocks                         */
/*                                                                            */
/* @brief Transforms and analyzes a string command into a useful command.     */
/* @param [in] cmdstr Blocks of strings representing the received command     */
/* @param [out] pCmd  Command built from the string command                   */
/* @req SYS_REQ-0202-001 : Réception des commandes d’administration           */
/* @req SYS_REQ-0203-001 : Décodage d’une commande d’administration           */
/******************************************************************************/
static void __COM_Messages_RetrieveCommandFromBlocks
(
    const char cmdstr[C_RSADMININ_MAX_BLOCK_NB][C_RSADMININ_MAX_BLOCK_LENGTH],
    tRsAdmin_Cmd * const pCmd
)
{
    __COM_Messages_Init_RSADMIN_Cmd(pCmd);

    if     (strcmp(cmdstr[__C_RSADMIN_FIELD_IDT_POS], "PING") == 0) { __COM_Messages_Decode_RSADMIN_PING(cmdstr, pCmd); }
    else if(strcmp(cmdstr[__C_RSADMIN_FIELD_IDT_POS], "CALD") == 0) { __COM_Messages_Decode_RSADMIN_CALD(cmdstr, pCmd); }
    else if(strcmp(cmdstr[__C_RSADMIN_FIELD_IDT_POS], "TMSP") == 0) { __COM_Messages_Decode_RSADMIN_TMSP(cmdstr, pCmd); }
    else { pCmd->validity = E_COM_VALIDITY_UNKNOWN_MSG; }
}

/******************************************************************************/
/* @function __COM_Messages_ParseRawCommands                                  */
/*                                                                            */
/* @brief Decodes raw commands received on RS_ADMIN_IN interface.             */
/* @param [in] rawdata Raw data received from RS_ADMIN_IN                     */
/* @param [out] commands List of commands extracted from raw data             */
/* @retval Number of commands found                                           */
/* @req SYS_REQ-0202-001 : Réception des commandes d’administration > Parsing */
/*                                                                            */
/* Commands are divided into blocks as following :                            */
/* - Maximum C_RSADMININ_MAX_CMDS commands                                    */
/* - Maximum C_RSADMININ_MAX_BLOCK_NB blocks in a command (blocks are         */
/*   separated by commas)                                                     */
/* - Maximum C_RSADMININ_MAX_BLOCK_LENGTH characters in a block               */
/******************************************************************************/
static unsigned int __COM_Messages_ParseRawCommands
(
    const tRsAdminBuffer rawdata,
    char commands[C_RSADMININ_MAX_CMDS][C_RSADMININ_MAX_BLOCK_NB][C_RSADMININ_MAX_BLOCK_LENGTH]
)
{
    unsigned int rawc;   /* Raw data characters iterator            */
    unsigned int nbCmds; /* Number of commands detected             */
    unsigned int blck;   /* Current block in the command            */
    unsigned int subc;   /* Character iterator in the current block */
    bool commandStartCharFound; /* Command starting character found */

    memset(commands, 0, C_RSADMININ_MAX_CMDS * C_RSADMININ_MAX_LENGTH * sizeof(char));

    nbCmds = 0;
    blck = __C_RSADMIN_FIELD_IDT_POS;
    subc = 0;
    commandStartCharFound = false;
    for(rawc = 0 ; rawc < C_RSADMIN_RECEIVE_BUFFER_MAX_LENGTH ; ++rawc)
    {
        switch(rawdata[rawc])
        {
            case '\r':
            case '\n':
            case '\0':
                /* ---------------------------------------------------------- */
                /* If the character is an end of line or end of file, there   */
                /* are two possibilities:                                     */
                /* - The previous message was a valid command whose ending    */
                /*   has been detected and the command has been saved (moved  */
                /*   to next command).                                        */
                /* - The previous message is not finished, and the end of     */
                /*   line or file indicates that the message is incomplete or */
                /*   corrupted.                                               */
                /* In both cases, either the previous command was saved and   */
                /* there is nothing to do, either the previous message is     */
                /* corrupted and shall be dropped and overwritten by new data.*/
                /* ---------------------------------------------------------- */
                if(blck != __C_RSADMIN_FIELD_IDT_POS || subc != 0)
                {
                    memset(commands[nbCmds], 0, C_RSADMININ_MAX_BLOCK_NB * C_RSADMININ_MAX_BLOCK_LENGTH * sizeof(char));
                    blck = __C_RSADMIN_FIELD_IDT_POS;
                    subc = 0;
                }
                commandStartCharFound = false;
                break;

            case __C_RSADMIN_MESSAGE_SEPARATOR_START:
                /* ---------------------------------------------------------- */
                /* If the character is a command beginning character, either  */
                /* the current command is empty (end of previous command      */
                /* previously detected), either the previous characters saved */
                /* are a troncated command or corrupted message.              */
                /* In both cases, if a beginning character is detected, the   */
                /* current message is dropped and overwritten with the new    */
                /* data from this separator.                                  */
                /* ---------------------------------------------------------- */
                if(blck != __C_RSADMIN_FIELD_IDT_POS || subc != 0)
                {
                    memset(commands[nbCmds], 0, C_RSADMININ_MAX_BLOCK_NB * C_RSADMININ_MAX_BLOCK_LENGTH * sizeof(char));
                    blck = __C_RSADMIN_FIELD_IDT_POS;
                    subc = 0;
                }
                commandStartCharFound = true;
                break;

            case __C_RSADMIN_MESSAGE_SEPARATOR_BLOCK:
                /* ---------------------------------------------------------- */
                /* If the character is a command separator character, moves   */
                /* to the next block.                                         */
                /* ---------------------------------------------------------- */
                if(commandStartCharFound)
                {
                    blck += 1;
                    subc  = 0;
                }
                /* ---------------------------------------------------------- */
                /* If the next block is out of the command (its index is      */
                /* greater than the number of blocks autorized) then the      */
                /* message is dropped and overwritten with next data.         */
                /* ---------------------------------------------------------- */
                if(blck >= C_RSADMININ_MAX_BLOCK_NB)
                {
                    memset(commands[nbCmds], 0, C_RSADMININ_MAX_BLOCK_NB * C_RSADMININ_MAX_BLOCK_LENGTH * sizeof(char));
                    blck = __C_RSADMIN_FIELD_IDT_POS;
                    subc = 0;
                    commandStartCharFound = false;
                }
                break;

            case __C_RSADMIN_MESSAGE_SEPARATOR_END:
                /* ---------------------------------------------------------- */
                /* If the character is a command ending character and if      */
                /* content has been found for the current command, moves to   */
                /* the next command.                                          */
                /* ---------------------------------------------------------- */
                if((blck != __C_RSADMIN_FIELD_IDT_POS || subc != 0) && commandStartCharFound)
                {
                    nbCmds += 1;
                    blck   = __C_RSADMIN_FIELD_IDT_POS;
                    subc   = 0;
                    commandStartCharFound = false;
                }
                break;

            default:
                /* ---------------------------------------------------------- */
                /* Other characters are command data and shall be saved in    */
                /* the current command.                                       */
                /* ---------------------------------------------------------- */
                if(commandStartCharFound)
                {
                    if(subc >= C_RSADMININ_MAX_BLOCK_LENGTH)
                    {
                        memset(commands[nbCmds], 0, C_RSADMININ_MAX_BLOCK_NB * C_RSADMININ_MAX_BLOCK_LENGTH * sizeof(char));
                        blck = __C_RSADMIN_FIELD_IDT_POS;
                        subc = 0;
                        commandStartCharFound = false;
                    }
                    else
                    {
                        commands[nbCmds][blck][subc] = rawdata[rawc];
                        subc += 1;
                    }
                }
                break;
        }
    }
    return nbCmds;
}


/* ************************** FUNCTIONS DEFINITION ************************** */

/******************************************************************************/
/* @function SUP_RsAdmin_DecodeCommand                                        */
/*                                                                            */
/* @brief Decodes commands received on RS_ADMIN_IN interface.                 */
/* @param [in] rawdata Raw data received from RS_ADMIN_IN                     */
/* @param [out] cmds List of commands extracted from raw data                 */
/* @retval Number of commands found                                           */
/* @req SYS_REQ-0202-001 : Réception des commandes d’administration           */
/* @req SYS_REQ-0203-001 : Décodage d’une commande d’administration           */
/******************************************************************************/
unsigned int SUP_RsAdmin_DecodeCommand
(
    const tRsAdminBuffer rawdata,
    tRsAdmin_Cmd cmds[C_RSADMININ_MAX_CMDS]
)
{
    unsigned int c;
    unsigned int nbCommands;
    char commands[C_RSADMININ_MAX_CMDS][C_RSADMININ_MAX_BLOCK_NB][C_RSADMININ_MAX_BLOCK_LENGTH];

    /* ====================================================================== */
    /* If no command has been received, skips the computations.               */
    /* ====================================================================== */
    if(rawdata[0] == '\0') { return 0; }

    /* ====================================================================== */
    /* Otherwise, if data has been received, parses the data to extract the   */
    /* string describing each command.                                        */
    /* ====================================================================== */
    nbCommands = __COM_Messages_ParseRawCommands(rawdata, commands);

    /* ====================================================================== */
    /* Otherwise, if data has been received, parses the data to extract the   */
    /* string describing each command.                                        */
    /* ====================================================================== */
    for(c = 0 ; c < nbCommands ; ++c)
    {
        __COM_Messages_RetrieveCommandFromBlocks(commands[c], &(cmds[c]));
    }

    return nbCommands;
}

/******************************************************************************/
/* @function SUP_RsAdmin_SelectCommandsToApply                                */
/*                                                                            */
/* @brief Selects the RS_ADMIN commands to take into account.                 */
/* @param [in] nbCmdsReceived Number of RS_ADMIN commands received            */
/* @param [in|out] cmds RS_ADMIN commands received                            */
/* @req SYS_REQ-0204-001 : Sélection des commandes prises en compte           */
/******************************************************************************/
void SUP_RsAdmin_SelectCommandsToApply
(
    const unsigned int nbCmdsReceived,
    tRsAdmin_Cmd cmds[C_RSADMININ_MAX_CMDS]
)
{
    int c;
    bool alreadyTakenIntoAccount[E_COM_RSADMIN_NB_CODES];

    /* ====================================================================== */
    /* For command codes other than PING, when multiple administration        */
    /* commands are received, only the last valid one shall be taken into     */
    /* account. In that case, the CMD::USED field of this command shall be    */
    /* set to TRUE. The CMD::USED field of other commands of the same code    */
    /* shall be set to FALSE.                                                 */
    /* The CMD::USED field of valid PING commands must always be set to TRUE. */
    /* ====================================================================== */
    for(c = nbCmdsReceived - 1 ; c >= 0 ; --c)
    {
        cmds[c].used = false;
        if(cmds[c].validity == E_COM_CMD_OK)
        {
            if(cmds[c].code == E_COM_RSADMIN_CODE_PING)
            {
                cmds[c].used = true;
            }
            else if(!alreadyTakenIntoAccount[cmds[c].code])
            {
                cmds[c].used = true;
                alreadyTakenIntoAccount[cmds[c].code] = true;
            }
        }
    }
}

/******************************************************************************/
/*                                                                            */
/*                             WorldSkills France                             */
/*                       48th edition - Marseille 2025                        */
/*                       Electronique (16) - SyncOrSink                       */
/*                                                                            */
/******************************************************************************/
