/******************************************************************************/
/*                                                                            */
/*                             WorldSkills France                             */
/*                              Electronique (16)                             */
/*                                                                            */
/******************************************************************************/
/* @file ADA1655.c                                                            */
/* @authors WorldSkills France "Electronique" skill team                      */
/* @version 1.1                                                               */
/* @date 2025-07-11                                                           */
/*                                                                            */
/* @brief ADA1655 driver abstraction layer.                                   */
/*        This file contains functions to interact with Adafruit ADA1655 led  */
/*        component.                                                          */
/*        WARNING : THIS CODE IS CALIBRATED TO WORK WITH A SYSCLK CLOCK SET   */
/*        TO 24MHz.                                                           */
/******************************************************************************/

#include <stddef.h>
#include "ADA1655.h"

#define __ADA1655_SHIFT_R  8
#define __ADA1655_SHIFT_G 16
#define __ADA1655_SHIFT_B  0
#define __ADA1655_TOTAL_S 24

/******************************************************************************/
/* @function ADA1655_SetControlPin                                            */
/*                                                                            */
/* @brief Selects the GPIO port and pin connected to an ADA1655 component.    */
/* @param [out] pAda1655  ADA1655 handler to set.                             */
/* @param [in]  GPIO_Port GPIO port linked to the ADA1655 led.                */
/* @param [in]  GPIO_Pin  GPIO pin connected to the ADA1655 led.              */
/* @retval This function returns nothing.                                     */
/******************************************************************************/
void ADA1655_SetControlPin(tAda1655 * const pAda1655, GPIO_TypeDef * const GPIO_Port, const uint16_t GPIO_Pin)
{
    pAda1655->gpio_port = GPIO_Port;
    pAda1655->gpio_pin  = GPIO_Pin;
}

/******************************************************************************/
/* @function ADA1655_SetColor                                                 */
/*                                                                            */
/* @brief Sets the ADA1655 light color.                                       */
/* @param [in] pAda1655  ADA1655 handler.                                     */
/* @param [in] r         "red"   light color component.                       */
/* @param [in] g         "green" light color component.                       */
/* @param [in] b         "blue"  light color component.                       */
/* @retval This function returns nothing.                                     */
/* @pre The ADA1655_SetControlPin function must have been called prior to     */
/*      this function call.                                                   */
/******************************************************************************/
void ADA1655_SetColor(const tAda1655 * const pAda1655, const uint8_t r, const uint8_t g, const uint8_t b)
{
    uint32_t data;
    int bitToSend;

    /* ---------------------------------------------------------------------- */
    /* Color code is sent to ADA1655 in the following order:                  */
    /* - Green color code, from MSB (G7) to LSB (G0)                          */
    /* - Red   color code, from MSB (R7) to LSB (R0)                          */
    /* - Blue  color code, from MSB (B7) to LSB (B0)                          */
    /* A total of 24 bits (3 x 8 bits) is sent.                               */
    /* ---------------------------------------------------------------------- */

    data = 0;
    data |= ((uint32_t) r << __ADA1655_SHIFT_R);
    data |= ((uint32_t) g << __ADA1655_SHIFT_G);
    data |= ((uint32_t) b << __ADA1655_SHIFT_B);
    bitToSend = __ADA1655_TOTAL_S - 1; /* 24 bits to send : 0..23 */

    /* ---------------------------------------------------------------------- */
    /* The ADA1655 data transmission protocol uses a unipolar zeroing code.   */
    /* Each symbol must have both high and low levels: in this protocol, each */
    /* each symbol starts with a high level, then is pulled to low level.     */
    /* The high time duration determines the binary value (0 or 1):           */
    /* - To encode "0", the high state (T0H) must last at least 0.20µs and    */
    /*   less than 0.40µs. The low state (T0L) must last at least 0.80µs.     */
    /* - To encode "1", the high state (T1H) must last at least 0.65µs and    */
    /*   less than 1.00µs. The low state (T1L) must last at least 0.20µs.     */
    /* ---------------------------------------------------------------------- */

    /* Disabling interruptions to ensure transmission timings are respected.  */
    __disable_irq();

    /* As long as there are bits left to transmit, writes high and low levels */
    /* on the pin linked to ADA1655.                                          */
    while (bitToSend >= 0)
    {
        if ((data & (1 << bitToSend)) != 0)
        {
            /* Send T1H */
            pAda1655->gpio_port->BSRR = pAda1655->gpio_pin;
            __NOP();
            __NOP();
            __NOP();
            __NOP();
            __NOP();
            __NOP();
            __NOP();
            __NOP();
            /* Send T1L */
            pAda1655->gpio_port->BRR = pAda1655->gpio_pin;
        }
        else
        {
            /* Send T0H */
            pAda1655->gpio_port->BSRR = pAda1655->gpio_pin;
            /* Send T0L */
            pAda1655->gpio_port->BRR = pAda1655->gpio_pin;
            __NOP();
        }
        --bitToSend;
    }

    /* Enabling interruptions back.                                           */
    __enable_irq();
}

/******************************************************************************/
/* @function ADA1655_Reset                                                    */
/*                                                                            */
/* @brief Resets the ADA1655 pin state.                                       */
/* @param [in] pAda1655 ADA1655 handler.                                      */
/* @retval This function returns nothing.                                     */
/******************************************************************************/
void ADA1655_Reset(tAda1655 * const pAda1655)
{
    HAL_GPIO_WritePin(pAda1655->gpio_port, pAda1655->gpio_pin, GPIO_PIN_RESET);
    HAL_Delay(1);
}

/******************************************************************************/
/*                                                                            */
/*                             WorldSkills France                             */
/*                              Electronique (16)                             */
/*                                                                            */
/******************************************************************************/
