/******************************************************************************/
/*                                                                            */
/*                             WorldSkills France                             */
/*                              Electronique (16)                             */
/*                                                                            */
/******************************************************************************/
/* @file PushButton.c                                                         */
/* @authors WorldSkills France "Electronique" skill team                      */
/* @version 1.0                                                               */
/* @date 2025-07-18                                                           */
/*                                                                            */
/* @brief Push button driver abstraction layer.                               */
/*        This file contains functions to handle a push button.               */
/******************************************************************************/

#include <stddef.h>
#include "PushButton.h"

/******************************************************************************/
/* @function PushButton_setError                                              */
/*                                                                            */
/* @brief Sets an error for the push button.                                  */
/* @param [out] pPushButton Push button handler.                              */
/* @param [in]  error       Push button error.                                */
/* @retval This function returns nothing.                                     */
/******************************************************************************/
static void PushButton_setError(tPushButton * const pPushButton, const tPushButtonError error)
{
    switch(error)
    {
        case PBE_NO_ERROR:
            pPushButton->error  = PBE_NO_ERROR;
            break;

        case PBE_MODE_ERROR:
            pPushButton->mode   = PBM_UNDEFINED;
            pPushButton->error  = PBE_MODE_ERROR;
            break;

        case PBE_PSTATE_ERROR:
            pPushButton->pstate = PBPS_UNDEFINED;
            pPushButton->error  = PBE_PSTATE_ERROR;
            break;

        case PBE_LSTATE_ERROR:
            pPushButton->lstate = PBLS_UNDEFINED;
            pPushButton->error  = PBE_LSTATE_ERROR;
            break;

        case PBE_UNDEFINED_ERROR:
        default:
            pPushButton->error  = PBE_UNDEFINED_ERROR;
            break;
    }
}

/******************************************************************************/
/* @function PushButton_init                                                  */
/*                                                                            */
/* @brief Initializes the hardware and logical contexts of a push button.     */
/* @param [out] pPushButton Push button handler to set.                       */
/* @param [in]  GPIO_Port GPIO port linked to the push button.                */
/* @param [in]  GPIO_Pin  GPIO pin connected to the push button.              */
/* @param [in]  mode      Push button behavior mode.                          */
/* @retval This function returns nothing.                                     */
/******************************************************************************/
void PushButton_init
(
    tPushButton  * const  pPushButton,
    GPIO_TypeDef * const  GPIO_Port,
    const uint16_t        GPIO_Pin,
    const tPushButtonMode mode
)
{
    pPushButton->gpio_port = GPIO_Port;
    pPushButton->gpio_pin  = GPIO_Pin;
    pPushButton->error     = PBE_NO_ERROR;
    pPushButton->pstate    = PBPS_UNDEFINED;
    pPushButton->held      = false;
    pPushButton->mode      = mode;
    pPushButton->lstate    = PBLS_INACTIVE;
}

/******************************************************************************/
/* @function PushButton_read                                                  */
/*                                                                            */
/* @brief Reads the state of a push button (sets pPushButton::state and       */
/*        pPushButton::held).                                                 */
/* @param [in|out] pPushButton Push button handler.                           */
/* @retval 'true' if an error occured (pPushButton::error is then set);       */
/*         'false' otherwise (pPushButton::error set to PBE_NO_ERROR).        */
/*         If no error, sets pPushButton::state and pPushButton::held.        */
/* @pre The PushButton_SetControlPin function must have been called prior to  */
/*      this function call.                                                   */
/******************************************************************************/
bool PushButton_read(tPushButton * const pPushButton)
{
    GPIO_PinState pinstate;
    tPushButtonPhysicalState previousPstate = pPushButton->pstate;

    /* ---- Button physical state reading ----------------------------------- */
    pinstate = HAL_GPIO_ReadPin(pPushButton->gpio_port, pPushButton->gpio_pin);

    /* ---- Setting button logical state ------------------------------------ */
    switch(pPushButton->mode)
    {
        case PBM_PRESS:
            /* In "press" mode, the push button is considered active when it  */
            /* is pressed, and inactive when released.                        */
            pPushButton->lstate = ((pinstate == GPIO_PIN_SET) ? PBLS_ACTIVE : PBLS_INACTIVE);
            break;

        case PBM_RISING_EDGE:
            /* In "rising edge" mode, the push button is considered active on */
            /* rising edge, i.e. on the first reading when the button is      */
            /* pressed. If the button is kept pressed on the next reading, it */
            /* is considered inactive as if the button was released.          */
            /* To ensure that the edge detection has been taken into account  */
            /* by the operational instructions, the flag "edge_ack" shall be  */
            /* set by the user to indicate that the button pressure has been  */
            /* seen by the program. The flag is defined to ensure that high   */
            /* frequency button detections are well taken into account by low */
            /* frequency running programs.                                    */
            if(pinstate == GPIO_PIN_SET)
            {
                /* If the button is physically pressed */
                if(previousPstate == PBPS_PRESSED)
                {
                    /* If the button was already pressed on previous reading  */
                    if(pPushButton->edge_ack)
                    {
                        /* If the pressure has been taken into account by the */
                        /* program, the button is then considered inactive,   */
                        /* but the acknowledgment remains active as long as   */
                        /* the button is physically pressed, to ensure the    */
                        /* pressure is not detected a second time.            */
                        pPushButton->lstate = PBLS_INACTIVE;
                    }
                    else
                    {
                        /* If the pressure has not yet been taken into        */
                        /* account by the program, wait for the program to    */
                        /* take it into account before performing any other   */
                        /* action.                                            */
                    }
                }
                else
                {
                    /* If the button was released on previous reading, this   */
                    /* event is a rising edge. The button is then considered  */
                    /* active and not already seen by the program.            */
                    pPushButton->lstate   = PBLS_ACTIVE;
                    pPushButton->edge_ack = false;
                }
            }
            else
            {
                /* If the button is physically released */
                if(pPushButton->edge_ack)
                {
                    /* If the previous pressure has been taken into account,  */
                    /* reset the button status.                               */
                    pPushButton->lstate   = PBLS_INACTIVE;
                    pPushButton->edge_ack = false;
                }
                else
                {
                    /* This case may happen in two situations:                */
                    /* - If the button has been released but the pressure has */
                    /*   not yet been taken into account by the program, wait */
                    /*   for the program to take it into account before       */
                    /*   performing any other action.                         */
                    /* - If the button has been released for some time and if */
                    /*   the previous pressure has been taken into account by */
                    /*   the program, the button is now in idle state and     */
                    /*   waiting for a new pressure. In that case, nothing to */
                    /*   do until the button is pressed.                      */
                }
            }
            break;

        case PBM_SWITCH:
            /* In "switch" mode, the status changes when the button is        */
            /* pressed, and maintained until the next time the button is      */
            /* pressed.                                                       */
            if((pinstate != GPIO_PIN_RESET) && (pPushButton->held == false))
            {
                switch(pPushButton->lstate)
                {
                    case PBLS_INACTIVE:
                        pPushButton->lstate = PBLS_ACTIVE;
                        break;
                    case PBLS_ACTIVE:
                        pPushButton->lstate = PBLS_INACTIVE;
                        break;
                    default:
                        PushButton_setError(pPushButton, PBE_LSTATE_ERROR);
                        return true;
                }
            }
            break;

        default:
            PushButton_setError(pPushButton, PBE_MODE_ERROR);
            return true;
    }

    /* ---------------------------------------------------------------------- */
    /* If the button status has not changed since the previous reading        */
    /* (if the current status equals the previous status), the button is      */
    /* considered kept pressed (held) by the user. Otherwise, it is released. */
    /* ---------------------------------------------------------------------- */
    pPushButton->pstate = ((pinstate == GPIO_PIN_SET) ? PBPS_PRESSED : PBPS_RELEASED);
    pPushButton->held   = ((pPushButton->pstate == previousPstate) ? true : false);

    /* ---------------------------------------------------------------------- */

    PushButton_setError(pPushButton, PBE_NO_ERROR);
    return false;
}


/******************************************************************************/
/*                                                                            */
/*                             WorldSkills France                             */
/*                              Electronique (16)                             */
/*                                                                            */
/******************************************************************************/
