/******************************************************************************/
/*                                                                            */
/*                             WorldSkills France                             */
/*                              Electronique (16)                             */
/*                                                                            */
/******************************************************************************/
/* @file PushButton.h                                                         */
/* @authors WorldSkills France "Electronique" skill team                      */
/* @version 1.0                                                               */
/* @date 2025-07-18                                                           */
/*                                                                            */
/* @brief Push button driver abstraction layer.                               */
/*        This file contains functions to handle a push button.               */
/******************************************************************************/

#ifndef __DEF_H_WSFR_16_PUSHBUTTON__
#define __DEF_H_WSFR_16_PUSHBUTTON__

#include <stdbool.h>
#include "stm32l0xx_hal.h"

typedef enum
{
    PBPS_UNDEFINED = 0,
    PBPS_RELEASED, /* Push button released */
    PBPS_PRESSED   /* Push button pressed  */
}tPushButtonPhysicalState;

typedef enum
{
    PBLS_UNDEFINED = 0,
    PBLS_INACTIVE, /* Push button inactive */
    PBLS_ACTIVE    /* Push button active   */
}tPushButtonLogicalState;

typedef enum
{
    PBM_UNDEFINED = 0,

    /* Push button considered "active" when pressed; "inactive" when released */
    PBM_PRESS,

    /* Push button considered "active" on rising edge (first reading when     */
    /* pressed), then considered "inactive" until the next time the button is */
    /* released then pressed, even if the button is kept pressed.             */
    PBM_RISING_EDGE,

    /* Push button considered "active" when pressed then released ;           */
    /* "inactive" when pressed then released again.                           */
    PBM_SWITCH
}tPushButtonMode;

typedef enum
{
    PBE_NO_ERROR = 0,
    PBE_MODE_ERROR,
    PBE_PSTATE_ERROR,
    PBE_LSTATE_ERROR,
    PBE_UNDEFINED_ERROR
}tPushButtonError;

typedef struct
{
    /* ---- Physical data acquisition --------------------------------------- */
    GPIO_TypeDef * gpio_port; /* Push button GPIO port                        */
    uint16_t gpio_pin;        /* Push button GPIO pin                         */
    tPushButtonError error;   /* Error encountered during button reading      */

    /* ---- Push button physical pressure state ----------------------------- */
    tPushButtonPhysicalState pstate; /* Push button physical pressure state   */
    bool held; /* True = same pstate as previous reading                      */

    /* ---- Push button logical state --------------------------------------- */
    tPushButtonMode mode;            /* Push button logical behavior mode     */
    tPushButtonLogicalState lstate;  /* Push button logical state             */

    /* For "edge" modes, the "edge_ack" flag shall explicitly be set by the   */
    /* user to acquit that the edge detection has been taken into account.    */
    /* This flag is defined to ensure taking into account button pressing if  */
    /* the pressure is detected at higher frequency than the instructions are */
    /* performed.                                                             */
    bool edge_ack;
}tPushButton;

void PushButton_init
(
    tPushButton  * const  pPushButton,
    GPIO_TypeDef * const  GPIO_Port,
    const uint16_t        GPIO_Pin,
    const tPushButtonMode mode
);

bool PushButton_read(tPushButton * const pPushButton);

#endif /* __DEF_H_WSFR_16_PUSHBUTTON__ */

/******************************************************************************/
/*                                                                            */
/*                             WorldSkills France                             */
/*                              Electronique (16)                             */
/*                                                                            */
/******************************************************************************/
