/******************************************************************************/
/*                                                                            */
/*                             WorldSkills France                             */
/*                       48th edition - Marseille 2025                        */
/*                       Electronique (16) - SyncOrSink                       */
/*                                                                            */
/******************************************************************************/
/* @file COM/COM_RfCarrier.c                                                  */
/* @authors WorldSkills France "Electronique" skill team                      */
/* @version 1.0                                                               */
/* @date 2025-09-07                                                           */
/*                                                                            */
/* @brief This file contains functions to control the RF transmission carrier.*/
/******************************************************************************/

/* *************************** STANDARD INCLUDES **************************** */

/* **************************** CUSTOM INCLUDES ***************************** */

#include "COM_RfCarrier.h"
#include "main.h"

/* ******************************** DEFINES ********************************* */

/* EPROM address where the modulation factor is stored:                       */
/* - @ addr 0x72 : value = 0x34 -> 1V signal out of R/2R DAC                  */
/* - @ addr 0x73 : value = 0xEF -> 5V signal out of R/2R DAC                  */
#define __C_COM_EPROM_ADDR_MODULATION_FACTOR (0x72)

/* *************************** TYPES DECLARATION **************************** */

typedef enum
{
    E_COM_EPROM_ADDR_BIT_00 = 0,
    E_COM_EPROM_ADDR_BIT_01,
    E_COM_EPROM_ADDR_BIT_02,
    E_COM_EPROM_ADDR_BIT_03,
    E_COM_EPROM_ADDR_BIT_04,
    E_COM_EPROM_ADDR_BIT_05,
    E_COM_EPROM_ADDR_BIT_06,
    E_COM_EPROM_ADDR_BIT_07,
    E_COM_EPROM_ADDR_NB_BITS
} tCOM_EpromBits;

typedef struct
{
    tCOM_EpromBits bit;
    GPIO_TypeDef * gpioPort;
    uint16_t       gpioPin;
} tCOM_EpromAddrBit;

/* ******************************* CONSTANTS ******************************** */

const tCOM_EpromAddrBit C_COM_EPROM_ADDR[E_COM_EPROM_ADDR_NB_BITS] =
{
    {E_COM_EPROM_ADDR_BIT_00, NULL                     , 0                  },
    {E_COM_EPROM_ADDR_BIT_01, COM_ADDR_BIT_01_GPIO_Port, COM_ADDR_BIT_01_Pin},
    {E_COM_EPROM_ADDR_BIT_02, COM_ADDR_BIT_02_GPIO_Port, COM_ADDR_BIT_02_Pin},
    {E_COM_EPROM_ADDR_BIT_03, COM_ADDR_BIT_03_GPIO_Port, COM_ADDR_BIT_03_Pin},
    {E_COM_EPROM_ADDR_BIT_04, COM_ADDR_BIT_04_GPIO_Port, COM_ADDR_BIT_04_Pin},
    {E_COM_EPROM_ADDR_BIT_05, COM_ADDR_BIT_05_GPIO_Port, COM_ADDR_BIT_05_Pin},
    {E_COM_EPROM_ADDR_BIT_06, COM_ADDR_BIT_06_GPIO_Port, COM_ADDR_BIT_06_Pin},
    {E_COM_EPROM_ADDR_BIT_07, COM_ADDR_BIT_07_GPIO_Port, COM_ADDR_BIT_07_Pin}
};

/* **************************** GLOBAL VARIABLES **************************** */

/* **************************** STATIC VARIABLES **************************** */

/* ********************** STATIC FUNCTIONS DECLARATION ********************** */

/* ************************* FUNCTIONS DECLARATION ************************** */

/* ********************** STATIC FUNCTIONS DEFINITION *********************** */

/* ************************** FUNCTIONS DEFINITION ************************** */

/******************************************************************************/
/* @function COM_ManageRfCarrier                                              */
/*                                                                            */
/* @brief Manages the RF transmission carrier.                                */
/* @param [in] rfCarrierActivationStatus RF carrier activation command        */
/* @pre The RF carrier control context must have been initialized by          */
/*      COM_RfCarrier_Init                                                    */
/* @req SYS_REQ-0603-001 : Redressage du signal RS_DATA                       */
/* @req SYS_REQ-0604-001 : Génération de la porteuse                          */
/******************************************************************************/
void COM_ManageRfCarrier(const bool rfCarrierActivationStatus)
{
    tCOM_EpromBits bit;

    if(rfCarrierActivationStatus)
    {
        /* ------------------------------------------------------------------ */
        /* If the RF carrier is activated:                                    */
        /* - The RF carrier is generated by activating the NAND-gates astable */
        /*   (SYS_REQ-0604-001)                                               */
        /* - The RS_DATA signal is rectified from [0;3.3]V to [1;5]V by       */
        /*   using the modulation factor stored in EPROM (SYS_REQ-0603-001).  */
        /* ------------------------------------------------------------------ */

        HAL_GPIO_WritePin(RF_CARRIER_ACTIVE_GPIO_Port, RF_CARRIER_ACTIVE_Pin, GPIO_PIN_SET);

        /* ------------------------------------------------------------------ */
        /* Note: The modulation factors stored in EPROM are placed one after  */
        /* the other. The __C_COM_EPROM_ADDR_MODULATION_FACTOR address stores */
        /* the low state value, and __C_COM_EPROM_ADDR_MODULATION_FACTOR + 1  */
        /* stores the high state value.                                       */
        /* The address COM_ADDR_BIT_00 is therefore never directly written by */
        /* this function, but is set by the RS_DATA signal.                   */
        /* ------------------------------------------------------------------ */

        for(bit = E_COM_EPROM_ADDR_BIT_01 ; bit < E_COM_EPROM_ADDR_NB_BITS ; ++bit)
        {
            HAL_GPIO_WritePin
            (
                C_COM_EPROM_ADDR[bit].gpioPort,
                C_COM_EPROM_ADDR[bit].gpioPin,
                (__C_COM_EPROM_ADDR_MODULATION_FACTOR >> bit) & 0x01
            );
        }
    }
    else
    {
        HAL_GPIO_WritePin(RF_CARRIER_ACTIVE_GPIO_Port, RF_CARRIER_ACTIVE_Pin, GPIO_PIN_RESET);

        for(bit = E_COM_EPROM_ADDR_BIT_01 ; bit < E_COM_EPROM_ADDR_NB_BITS ; ++bit)
        {
            HAL_GPIO_WritePin
            (
                C_COM_EPROM_ADDR[bit].gpioPort,
                C_COM_EPROM_ADDR[bit].gpioPin,
                GPIO_PIN_RESET
            );
        }
    }
}

/******************************************************************************/
/*                                                                            */
/*                             WorldSkills France                             */
/*                       48th edition - Marseille 2025                        */
/*                       Electronique (16) - SyncOrSink                       */
/*                                                                            */
/******************************************************************************/
