/******************************************************************************/
/*                                                                            */
/*                             WorldSkills France                             */
/*                       48th edition - Marseille 2025                        */
/*                       Electronique (16) - SyncOrSink                       */
/*                                                                            */
/******************************************************************************/
/* @file COM/COM_Messages_RsData.c                                            */
/* @authors WorldSkills France "Electronique" skill team                      */
/* @version 1.0                                                               */
/* @date 2025-09-12                                                           */
/*                                                                            */
/* @brief This file contains functions to encode RS_DATA messages.            */
/******************************************************************************/

/* *************************** STANDARD INCLUDES **************************** */

#include <stdio.h>
#include <string.h>
#include <inttypes.h>

/* **************************** CUSTOM INCLUDES ***************************** */

#include "COM_RsData.h"

/* ******************************** DEFINES ********************************* */

#define __C_RSDATA_FIELD_TSP_SIZE (14)

/* *************************** TYPES DECLARATION **************************** */

/* ******************************* CONSTANTS ******************************** */

/* **************************** GLOBAL VARIABLES **************************** */

/* **************************** STATIC VARIABLES **************************** */

/* ********************** STATIC FUNCTIONS DECLARATION ********************** */

/* ************************* FUNCTIONS DECLARATION ************************** */

/* ********************** STATIC FUNCTIONS DEFINITION *********************** */

/******************************************************************************/
/* @function i64_to_str                                                       */
/* @brief Converts a 64-bit integer into a string                             */
/* @author ChatGPT, as %lld seems not to work on STM32.                       */
/******************************************************************************/
static char *i64_to_str(int64_t value, char *buf, size_t bufsize) {
    if (bufsize == 0) return buf;
    char tmp[32];
    int pos = 0;
    uint64_t v;
    if (value < 0) {
        // conversion sans overflow pour INT64_MIN
        v = (uint64_t)(-(value + 1)) + 1;
    } else {
        v = (uint64_t)value;
    }
    if (v == 0) {
        if (bufsize < 2) { buf[0]=0; return buf; }
        buf[0] = '0'; buf[1] = '\0';
        return buf;
    }
    while (v && pos < (int)sizeof(tmp)-1) {
        tmp[pos++] = '0' + (v % 10);
        v /= 10;
    }
    size_t out = 0;
    if (value < 0) {
        if (out < bufsize - 1) buf[out++] = '-';
        else { buf[0]=0; return buf; }
    }
    while (pos > 0) {
        if (out < bufsize - 1) buf[out++] = tmp[--pos];
        else { buf[0]=0; return buf; }
    }
    buf[out] = '\0';
    return buf;
}

/******************************************************************************/
/* @function __COM_Messages_Build_RSDATA_PONG                                 */
/*                                                                            */
/* @brief Builds RS_DATA "PONG" message.                                      */
/* @param [in] pingIds Identifiers of received PING commands                  */
/* @param [in|out] buffer RS_DATA content to transmit                         */
/* @req SYS_REQ-0601-001 : Construction des trames RS_DATA > PONG             */
/******************************************************************************/
static void __COM_Messages_Build_RSDATA_PONG
(
    const tPingIds pingIds,
    tRsDataBuffer buffer
)
{
    char tmp[C_COM_TRANSMIT_BUFFER_MAX_LENGTH];
    unsigned int idx = 0;
    while((pingIds[idx][0] != '\0') && (idx < C_RSADMININ_PING_MAX_NB))
    {
        snprintf(tmp, C_COM_TRANSMIT_BUFFER_MAX_LENGTH, "$PONG,%s;", pingIds[idx]);
        strcat(buffer, tmp);
        ++idx;
    }
}

/******************************************************************************/
/* @function __COM_Messages_Build_RSDATA_ACKC                                 */
/*                                                                            */
/* @brief Builds RS_DATA "ACKC" message.                                      */
/* @param [in] newCalendar New calendar received by CALD admin command        */
/* @param [in|out] buffer RS_DATA content to transmit                         */
/* @req SYS_REQ-0601-001 : Construction des trames RS_DATA > ACKC             */
/******************************************************************************/
static void __COM_Messages_Build_RSDATA_ACKC
(
    const tTIME_Calendar newCalendar,
    tRsDataBuffer buffer
)
{
    char tmp[C_COM_TRANSMIT_BUFFER_MAX_LENGTH];

    switch(newCalendar)
    {
        case E_CALENDAR_UNKNOWN:
        case E_CALENDAR_JULIAN:
        case E_CALENDAR_GREGORIAN:
        case E_CALENDAR_UTC:
            snprintf(tmp, C_COM_TRANSMIT_BUFFER_MAX_LENGTH, "$ACKC,<id>,%s;", C_TIME_CALENDAR_SHRTNAME[newCalendar]);
            strcat(buffer, tmp);
            break;
        default:
            break;
    }
}

/******************************************************************************/
/* @function __COM_Messages_Build_RSDATA_ACKT                                 */
/*                                                                            */
/* @brief Builds RS_DATA "ACKT" message.                                      */
/* @param [in] currentCalendar Current calendar used by the system            */
/* @param [in] newTimestamp New timestamp received by TMSP admin command      */
/* @param [in|out] buffer RS_DATA content to transmit                         */
/* @req SYS_REQ-0601-001 : Construction des trames RS_DATA > ACKT             */
/******************************************************************************/
static void __COM_Messages_Build_RSDATA_ACKT
(
    const tTIME_Calendar currentCalendar,
    const tTIME_Timestamp newTimestamp,
    tRsDataBuffer buffer
)
{
    char tmp[C_COM_TRANSMIT_BUFFER_MAX_LENGTH];
    char tsp[__C_RSDATA_FIELD_TSP_SIZE];
    tTIME_CalendarDate date;

    if(newTimestamp != C_UNDEFINED_TIMESTAMP)
    {
        date = TIME_ComputeDate(currentCalendar, newTimestamp);
        i64_to_str(newTimestamp, tsp, __C_RSDATA_FIELD_TSP_SIZE);
        snprintf
        (
            tmp, C_COM_TRANSMIT_BUFFER_MAX_LENGTH,
            "$ACKT,<id>,%s,%s,%02i,%02i,%04i,%02i,%02i,%02i;",
            tsp, C_TIME_CALENDAR_SHRTNAME[currentCalendar],
            date.day + 1, date.month + 1, date.year,
            date.hour, date.minute, date.second
        );
        strcat(buffer, tmp);
    }
}

/* ************************** FUNCTIONS DEFINITION ************************** */

/******************************************************************************/
/* @function COM_Messages_Build_RSDATA                                        */
/*                                                                            */
/* @brief Formats messages for RS_DATA transmission.                          */
/* @param [in] rfCarrierActive RF carrier activation command                  */
/* @param [in] pingIds Identifiers of received PING commands                  */
/* @param [in] currentCalendar Current calendar used by the system            */
/* @param [in] newCalendar New calendar received by CALD admin command        */
/* @param [in] newTimestamp New timestamp received by TMSP admin command      */
/* @param [in|out] buffer RS_DATA content to transmit                         */
/* @req SYS_REQ-0601-001 : Construction des trames RS_DATA                    */
/******************************************************************************/
void COM_Messages_Build_RSDATA
(
    const bool rfCarrierActive,
    const tPingIds pingIds,
    const tTIME_Calendar currentCalendar,
    const tTIME_Calendar newCalendar,
    const tTIME_Timestamp newTimestamp,
    tRsDataBuffer buffer
)
{
    memset(buffer, 0, sizeof(tRsDataBuffer));
    if(rfCarrierActive)
    {
        __COM_Messages_Build_RSDATA_PONG(pingIds, buffer);
        __COM_Messages_Build_RSDATA_ACKC(newCalendar, buffer);
        __COM_Messages_Build_RSDATA_ACKT(currentCalendar, newTimestamp, buffer);
        if(buffer[0] != '\0') { strcat(buffer, "\n"); }
    }
}

/******************************************************************************/
/*                                                                            */
/*                             WorldSkills France                             */
/*                       48th edition - Marseille 2025                        */
/*                       Electronique (16) - SyncOrSink                       */
/*                                                                            */
/******************************************************************************/
