/******************************************************************************/
/*                                                                            */
/*                             WorldSkills France                             */
/*                       48th edition - Marseille 2025                        */
/*                       Electronique (16) - SyncOrSink                       */
/*                                                                            */
/******************************************************************************/
/* @file COM/COM_Transmission.c                                               */
/* @authors WorldSkills France "Electronique" skill team                      */
/* @version 1.0                                                               */
/* @date 2025-09-12                                                           */
/*                                                                            */
/* @brief This file contains functions to transmit data to remote equipments. */
/******************************************************************************/

/* *************************** STANDARD INCLUDES **************************** */

/* **************************** CUSTOM INCLUDES ***************************** */

#include "COM_Transmission.h"

/* ******************************** DEFINES ********************************* */

#define __C_COM_TRANSMISSION_TIMEOUT (1000) /* 1000 ms */

/* *************************** TYPES DECLARATION **************************** */

/* ******************************* CONSTANTS ******************************** */

/* **************************** GLOBAL VARIABLES **************************** */

/* **************************** STATIC VARIABLES **************************** */

static UART_HandleTypeDef * __COM_TRANSMISSION_UART_HANDLER;

/* ********************** STATIC FUNCTIONS DECLARATION ********************** */

/* ************************* FUNCTIONS DECLARATION ************************** */

/* ********************** STATIC FUNCTIONS DEFINITION *********************** */

/* ************************** FUNCTIONS DEFINITION ************************** */

/******************************************************************************/
/* @function COM_Transmission_Init                                            */
/*                                                                            */
/* @brief Initializes the transmission interface for remote equipments.       */
/* @param [in] UartHandler UART serial transmission line handler              */
/* @req SYS_REQ-0602-001 : Transmission des données RS_DATA                   */
/******************************************************************************/
void COM_Transmission_Init(UART_HandleTypeDef * const UartHandler)
{
    /* ====================================================================== */
    /* The system shall provide an unidirectional UART serial transmission    */
    /* interface to send data to remote equipments. This serial interface     */
    /* shall be configured as follows:                                        */
    /* - Baude rate : 9600 bauds                                              */
    /* - 8 data bits                                                          */
    /* - Odd parity bit                                                       */
    /* - 1 stop bit                                                           */
    /* ====================================================================== */
    /* The serial line configuration is generated automagically in the        */
    /* MX_USART1_UART_Init function.                                          */
    /* ---------------------------------------------------------------------- */
    __COM_TRANSMISSION_UART_HANDLER = UartHandler;
}

/******************************************************************************/
/* @function COM_Transmission_SendData                                        */
/*                                                                            */
/* @brief Sends data to remote equipment.                                     */
/* @param [in] buffer Content to send on RS_DATA transmission line            */
/* @pre Transmission context must have been initialized with                  */
/*      COM_Transmission_Init                                                 */
/* @req SYS_REQ-0602-001 : Transmission des données RS_DATA                   */
/******************************************************************************/
void COM_Transmission_SendData(const tRsDataBuffer buffer)
{
    unsigned int strl = 0;
    while(buffer[strl] != '\0' && strl < C_COM_TRANSMIT_BUFFER_MAX_LENGTH)
    {
        strl += 1;
    }
    HAL_UART_Transmit(__COM_TRANSMISSION_UART_HANDLER, (const unsigned char *)buffer, strl, __C_COM_TRANSMISSION_TIMEOUT);
}

/******************************************************************************/
/*                                                                            */
/*                             WorldSkills France                             */
/*                       48th edition - Marseille 2025                        */
/*                       Electronique (16) - SyncOrSink                       */
/*                                                                            */
/******************************************************************************/
