/******************************************************************************/
/*                                                                            */
/*                             WorldSkills France                             */
/*                       48th edition - Marseille 2025                        */
/*                       Electronique (16) - SyncOrSink                       */
/*                                                                            */
/******************************************************************************/
/* @file PWR/PWR.c                                                            */
/* @authors WorldSkills France "Electronique" skill team                      */
/* @version 1.0                                                               */
/* @date 2025-09-06                                                           */
/*                                                                            */
/* @brief This file contains functions to monitor system power supply.        */
/******************************************************************************/

/* *************************** STANDARD INCLUDES **************************** */

#include <math.h>

/* **************************** CUSTOM INCLUDES ***************************** */

#include "PWR.h"

/* ******************************** DEFINES ********************************* */

#define C_PWR_ADC_MAX_VALUE ((double)4095.0) /* 12-bit ADC */
#define C_PWR_PS_EXT_P_MAX_VOLTAGE ((double) 9.0) /* PS_EXT_P = [ 0 ; 9] V */
#define C_PWR_PS_EXT_N_MAX_VOLTAGE ((double)-9.0) /* PS_EXT_N = [-9 ; 0] V */

/* *************************** TYPES DECLARATION **************************** */

/* ******************************* CONSTANTS ******************************** */

/* **************************** GLOBAL VARIABLES **************************** */

/* **************************** STATIC VARIABLES **************************** */

static unsigned int __PWR_POWERSOURCE_ADC_CHANNELS[E_NB_POWERSOURCE];

static ADC_HandleTypeDef * __PWR_ADC_HANDLER;

/* Warning: DMA is configured in half-word values (16 bit values) */
static volatile uint16_t __PWR_ADC_DMA_VALUES[E_NB_POWERSOURCE];

/* ********************** STATIC FUNCTIONS DECLARATION ********************** */

/* ************************* FUNCTIONS DECLARATION ************************** */

/* ********************** STATIC FUNCTIONS DEFINITION *********************** */

/* ************************** FUNCTIONS DEFINITION ************************** */

/******************************************************************************/
/* @function PWR_ReadPowerSourceVoltage                                       */
/*                                                                            */
/* @brief Reads the voltage of a board power supply.                          */
/* @param [in] powerSource Power supply source to read                        */
/* @retval Power supply voltage if the power source is known; NaN otherwise.  */
/* @req SYS_REQ-0106-001 : Numérisation de la tension d'alimentation positive */
/* @req SYS_REQ-0107-001 : Numérisation de la tension d'alimentation négative */
/******************************************************************************/
double PWR_ReadPowerSourceVoltage(const tPWR_PowerSource powerSource)
{
    double rawAdcValue;
    double powerSourceVoltage;

    /* ---------------------------------------------------------------------- */
    /* Note: as the DMA directly copies the ADC values into RAM in a          */
    /* user-defined variable (here __PWR_ADC_DMA_VALUES), there is a risk     */
    /* that the program reads the values at the same time they are written,   */
    /* which may lead to inconsistent data read. To prevent this behavior,    */
    /* the HAL_ADC_ConvCpltCallback callback function can be defined to store */
    /* the values in non volatile memory upon DMA copy. However, there is no  */
    /* need for precise reading in this application and an occasional reading */
    /* error is acceptable, so this method is not implemented.                */
    /* ---------------------------------------------------------------------- */
    /* Note: the ADC sampling duration is not guaranteed, so the data stored  */
    /* in __PWR_ADC_RAW_VALUES may not always be up to date.                  */
    /* To ensure periodic ADC reading, a method might be to poll ADC values   */
    /* during a periodic interruption instead of asking the hardware to do    */
    /* the job on its own. However, there is no need for precise reading in   */
    /* this application, so this method is not implemented.                   */
    /* ---------------------------------------------------------------------- */
    switch(powerSource)
    {
        case E_POWERSOURCE_P:
            /* PS_EXT_P = [0;9] V converted to PS_EXT_P_Voltage = [0;3.3] V   */
            /* in order to be numerized by the ADC. As the [0;9]->[0;3.3]     */
            /* value mapping is linear, and as 3.3V is the maximum value the  */
            /* ADC can read, the [0;3.3] mapping is transparent for the       */
            /* ADC raw value to [0;9] voltage conversion.                     */
            rawAdcValue = (double)__PWR_ADC_DMA_VALUES[E_POWERSOURCE_P];
            powerSourceVoltage = (rawAdcValue / C_PWR_ADC_MAX_VALUE) * C_PWR_PS_EXT_P_MAX_VOLTAGE;
            break;

        case E_POWERSOURCE_N:
            /* PS_EXT_N = [-9;0] V converted to PS_EXT_N_Voltage = [0;3.3] V  */
            /* in order to be numerized by the ADC. As the [-9;0]->[0;3.3]    */
            /* value mapping is symetric-linear, and as 3.3V is the maximum   */
            /* value the ADC can read, the [0;3.3] mapping is transparent for */
            /* the ADC raw value to [0;9] voltage conversion.                 */
            rawAdcValue = (double)__PWR_ADC_DMA_VALUES[E_POWERSOURCE_N];
            powerSourceVoltage = (rawAdcValue / C_PWR_ADC_MAX_VALUE) * C_PWR_PS_EXT_N_MAX_VOLTAGE;
            break;

        default:
            powerSourceVoltage = NAN;
            break;
    }

    return powerSourceVoltage;
}

/******************************************************************************/
/* @function PWR_Init                                                         */
/*                                                                            */
/* @brief Initializes power supply monitoring context.                        */
/* @param [in|out] adcHandler Analog to digital converter handler             */
/* @retval Result value of HAL_ADC_Start_DMA function.                        */
/* @req None                                                                  */
/******************************************************************************/
HAL_StatusTypeDef PWR_Init(ADC_HandleTypeDef * const adcHandler)
{
    tPWR_PowerSource ipwrs;
    HAL_StatusTypeDef retval;

    __PWR_ADC_HANDLER = adcHandler;

    __PWR_POWERSOURCE_ADC_CHANNELS[E_POWERSOURCE_P] = ADC_CHANNEL_0;
    __PWR_POWERSOURCE_ADC_CHANNELS[E_POWERSOURCE_N] = ADC_CHANNEL_1;

    /* ---------------------------------------------------------------------- */
    /* Power supply monitoring is done by reading power supply voltages.      */
    /* To do so, the ADC (analog to digital converter) is configured to read  */
    /* both positive (PS_P_Voltage, on channel ADC_CHANNEL_0) and negative    */
    /* (PS_N_Voltage, on channel ADC_CHANNEL_1) board power sources.          */
    /*                                                                        */
    /* There are two methods to read the ADC values :                         */
    /* - Polling method, which can only read one channel at a time and        */
    /*   requires to reconfigure ADC each time to change the channel read.    */
    /* - Continuous reading with DMA copy, which continuously reads the ADC   */
    /*   values and store them in an array.                                   */
    /*                                                                        */
    /* The DMA method is more complex than the polling method, but have many  */
    /* benefits:                                                              */
    /* - There is no need to reconfigure the ADC each time, which reduces CPU */
    /*   usage as less instructions are performed.                            */
    /* - The values are always available and there is no need to wait for the */
    /*   ADC to read them (the reading is done in background, while polling   */
    /*   method starts to read the values only when asked to).                */
    /* - The cyclic reading is a hardware feature, and DMA only copies values */
    /*   to RAM when they are ready, which means no CPU usage at all!         */
    /* ---------------------------------------------------------------------- */

    for(ipwrs = 0 ; ipwrs < E_NB_POWERSOURCE ; ++ipwrs)
    {
        __PWR_ADC_DMA_VALUES[ipwrs] = 0;
    }

    retval = HAL_ADC_Start_DMA(__PWR_ADC_HANDLER, (uint32_t*)__PWR_ADC_DMA_VALUES, E_NB_POWERSOURCE);
    return retval;
}

/******************************************************************************/
/*                                                                            */
/*                             WorldSkills France                             */
/*                       48th edition - Marseille 2025                        */
/*                       Electronique (16) - SyncOrSink                       */
/*                                                                            */
/******************************************************************************/
