/******************************************************************************/
/*                                                                            */
/*                             WorldSkills France                             */
/*                       48th edition - Marseille 2025                        */
/*                       Electronique (16) - SyncOrSink                       */
/*                                                                            */
/******************************************************************************/
/* @file SUP/SUP_RsAdmin.h                                                    */
/* @authors WorldSkills France "Electronique" skill team                      */
/* @version 1.0                                                               */
/* @date 2025-09-12                                                           */
/*                                                                            */
/* @brief This file contains functions to decode RS_ADMIN_IN messages.        */
/******************************************************************************/

#ifndef __DEF_H_WSFR48_16_SUP_RSADMIN__
#define __DEF_H_WSFR48_16_SUP_RSADMIN__

/* *************************** STANDARD INCLUDES **************************** */

#include <stdbool.h>

/* **************************** CUSTOM INCLUDES ***************************** */

/* ******************************* CONSTANTS ******************************** */

#define C_RSADMININ_LENGTH_IDT   (  4) /* Length of "IDT" field in RS_ADMIN messages              */
#define C_RSADMININ_LENGTH_NUM   (  2) /* Length of "NUM" field in RS_ADMIN messages              */
#define C_RSADMININ_LENGTH_DATA  ( 14) /* Maximum length of "DATA" field in RS_ADMIN messages     */
#define C_RSADMININ_LENGTH_SEP   (  8) /* Maximum length of added separators in RS_ADMIN messages */
#define C_RSADMININ_MAX_BLOCK_NB (  3) /* Maximum number of blocks in a RS_ADMIN_IN message       */
#define C_RSADMININ_MAX_BLOCK_LENGTH (C_RSADMININ_LENGTH_DATA) /* Maximum length of a block       */

/* RS_ADMIN message maximum length */
#define C_RSADMININ_MAX_LENGTH ( \
        C_RSADMININ_LENGTH_IDT   \
      + C_RSADMININ_LENGTH_NUM   \
      + C_RSADMININ_LENGTH_DATA  \
      + C_RSADMININ_LENGTH_SEP   \
      )
#define C_RSADMININ_MAX_CMDS     ( 32) /* Maximum number of commands admissible on one cycle */

/* RS_ADMIN reception buffer size */
#define C_RSADMIN_RECEIVE_BUFFER_MAX_LENGTH  (C_RSADMININ_MAX_CMDS * (C_RSADMININ_MAX_LENGTH + 2))

#define C_RSADMININ_PING_MAX_NB  ( 10) /* Maximum number of pings allowed during one computation cycle */

/* *************************** TYPES DECLARATION **************************** */

typedef char tRsAdminBuffer[C_RSADMIN_RECEIVE_BUFFER_MAX_LENGTH];
typedef char tCmdId[C_RSADMININ_LENGTH_NUM + 1];

typedef enum
{
    E_COM_RSADMIN_CODE_UNDEFINED = 0,
    E_COM_RSADMIN_CODE_PING,
    E_COM_RSADMIN_CODE_CALD,
    E_COM_RSADMIN_CODE_TMSP,
    E_COM_RSADMIN_NB_CODES
}tRsAdmin_Code;

typedef enum
{
    E_COM_VALIDITY_UNDEFINED = 0,
    E_COM_CMD_OK,
    E_COM_VALIDITY_UNKNOWN_MSG,
    E_COM_VALIDITY_FORMAT_ERROR,
    E_COM_VALIDITY_OUT_OF_RANGE,
    E_COM_NB_VALIDITY
}tRsAdmin_Validity;

typedef struct
{
    tRsAdmin_Validity validity; /* Command validity                           */
    bool used;                  /* TRUE = command shall be taken into account */
    tRsAdmin_Code code;         /* Type of command                            */
    tCmdId id;                  /* Command unique identifier                  */
    char data[C_RSADMININ_LENGTH_DATA];
}tRsAdmin_Cmd;

/* **************************** GLOBAL VARIABLES **************************** */

/* ************************* FUNCTIONS DECLARATION ************************** */

/******************************************************************************/
/* @function SUP_RsAdmin_DecodeCommand                                        */
/*                                                                            */
/* @brief Decodes commands received on RS_ADMIN_IN interface.                 */
/* @param [in] rawdata Raw data received from RS_ADMIN_IN                     */
/* @param [out] cmds List of commands extracted from raw data                 */
/* @retval Number of commands found                                           */
/* @req SYS_REQ-0202-001 : Réception des commandes d’administration           */
/* @req SYS_REQ-0203-001 : Décodage d’une commande d’administration           */
/******************************************************************************/
unsigned int SUP_RsAdmin_DecodeCommand
(
    const tRsAdminBuffer rawdata,
    tRsAdmin_Cmd cmds[C_RSADMININ_MAX_CMDS]
);

/******************************************************************************/
/* @function SUP_RsAdmin_SelectCommandsToApply                                */
/*                                                                            */
/* @brief Selects the RS_ADMIN commands to take into account.                 */
/* @param [in] nbCmdsReceived Number of RS_ADMIN commands received            */
/* @param [in|out] cmds RS_ADMIN commands received                            */
/* @req SYS_REQ-0204-001 : Sélection des commandes prises en compte           */
/******************************************************************************/
void SUP_RsAdmin_SelectCommandsToApply
(
    const unsigned int nbCmdsReceived,
    tRsAdmin_Cmd cmds[C_RSADMININ_MAX_CMDS]
);

/******************************************************************************/
/*                                                                            */
/*                             WorldSkills France                             */
/*                       48th edition - Marseille 2025                        */
/*                       Electronique (16) - SyncOrSink                       */
/*                                                                            */
/******************************************************************************/

#endif /* __DEF_H_WSFR48_16_SUP_RSADMIN__ */
