/******************************************************************************/
/*                                                                            */
/*                             WorldSkills France                             */
/*                       48th edition - Marseille 2025                        */
/*                       Electronique (16) - SyncOrSink                       */
/*                                                                            */
/******************************************************************************/
/* @file TIME/TIME.c                                                          */
/* @authors WorldSkills France "Electronique" skill team                      */
/* @version 1.0                                                               */
/* @date 2025-09-06                                                           */
/*                                                                            */
/* @brief This file contains functions to measure and compute date and time.  */
/******************************************************************************/

/* *************************** STANDARD INCLUDES **************************** */

#include <math.h>

/* **************************** CUSTOM INCLUDES ***************************** */

#include "TIME.h"

/* ******************************** DEFINES ********************************* */

/* *************************** TYPES DECLARATION **************************** */

/* ******************************* CONSTANTS ******************************** */

/* **************************** GLOBAL VARIABLES **************************** */

/* Number of milliseconds elapsed since software startup */
static tTIME_Timestamp __TIME_TIME_ELAPSED_SINCE_STARTUP;

/* User-defined reference timestamp (ms) */
static tTIME_Timestamp __TIME_REFERENCE_TIMESTAMP;

/* Number of milliseconds elapsed since startup when the reference timestamp  */
/* was set                                                                    */
static tTIME_Timestamp __TIME_TESS_AT_REFERENCE_TIMESTAMP;

/* **************************** STATIC VARIABLES **************************** */

/* ********************** STATIC FUNCTIONS DECLARATION ********************** */

/* ************************* FUNCTIONS DECLARATION ************************** */

/* ********************** STATIC FUNCTIONS DEFINITION *********************** */

/* ************************** FUNCTIONS DEFINITION ************************** */

/******************************************************************************/
/* @function TIME_GetTimeElapsedSinceStartup                                  */
/*                                                                            */
/* @brief Gets the number of milliseconds elapsed since system startup.       */
/* @retval Number of milliseconds elapsed since system startup.               */
/* @pre The time context must have been initialized by TIME_Init.             */
/* @req SYS_REQ-0501-002 : Temps écoulé depuis le démarrage du système        */
/******************************************************************************/
tTIME_Timestamp TIME_GetTimeElapsedSinceStartup(void)
{
    /* ====================================================================== */
    /* Time elapsed since startup is incremented by TIM21. TIM21 interrupt    */
    /* frequency is set by MX_TIM21_Init and TIME_ELAPSED_SINCE_STARTUP       */
    /* dataflow is incremented by TIM21_IRQHandler function.                  */
    /* TIM21 config is set as follows:                                        */
    /* - Fclk = 32 MHz                                                        */
    /* - PSC  = 0                                                             */
    /* - ARR  = 31999                                                         */
    /* Finterrupt = Fclk / ((PSC+1)(ARR+1)) = 32.10^6/(1*32000) = 1 kHz       */
    /*                                                          => T = 1 ms   */
    /* ====================================================================== */
    return __TIME_TIME_ELAPSED_SINCE_STARTUP;
}

/******************************************************************************/
/* @function TIME_IncrementTimeElapsedSinceStartup                            */
/*                                                                            */
/* @brief Increments the number of milliseconds elapsed since system startup  */
/*        by 1ms.                                                             */
/* @pre The time context must have been initialized by TIME_Init.             */
/* @req SYS_REQ-0501-002 : Temps écoulé depuis le démarrage du système        */
/******************************************************************************/
void TIME_IncrementTimeElapsedSinceStartup(void)
{
    ++__TIME_TIME_ELAPSED_SINCE_STARTUP;
}

/******************************************************************************/
/* @function TIME_GetTimestamp                                                */
/*                                                                            */
/* @brief Gets the current timestamp in seconds.                              */
/* @pre The time context must have been initialized by TIME_Init.             */
/* @req SYS_REQ-0502-001 : Calcul du timestamp actuel                         */
/******************************************************************************/
tTIME_Timestamp TIME_GetTimestamp(void)
{
    tTIME_Timestamp timestamp_ms;
    timestamp_ms = __TIME_REFERENCE_TIMESTAMP + (__TIME_TIME_ELAPSED_SINCE_STARTUP - __TIME_TESS_AT_REFERENCE_TIMESTAMP);
    return (tTIME_Timestamp) floor((double)timestamp_ms / C_NB_MS_IN_S);
}

/******************************************************************************/
/* @function TIME_SetReferenceTimestamp                                       */
/*                                                                            */
/* @brief Sets the reference timestamp.                                       */
/* @param [in] newTimestamp New reference timestamp, in milliseconds          */
/* @pre The time context must have been initialized by TIME_Init.             */
/* @req None                                                                  */
/******************************************************************************/
void TIME_SetReferenceTimestamp(const tTIME_Timestamp newTimestamp)
{
    __TIME_REFERENCE_TIMESTAMP = newTimestamp;
    __TIME_TESS_AT_REFERENCE_TIMESTAMP = __TIME_TIME_ELAPSED_SINCE_STARTUP;
}

/******************************************************************************/
/* @function TIME_Init                                                        */
/*                                                                            */
/* @brief Initializes the time context.                                       */
/* @req None                                                                  */
/******************************************************************************/
void TIME_Init(void)
{
    __TIME_TIME_ELAPSED_SINCE_STARTUP = 0;
    __TIME_REFERENCE_TIMESTAMP = 0;
    __TIME_TESS_AT_REFERENCE_TIMESTAMP = __TIME_TIME_ELAPSED_SINCE_STARTUP;
}

/******************************************************************************/
/*                                                                            */
/*                             WorldSkills France                             */
/*                       48th edition - Marseille 2025                        */
/*                       Electronique (16) - SyncOrSink                       */
/*                                                                            */
/******************************************************************************/
