/******************************************************************************/
/*                                                                            */
/*                             WorldSkills France                             */
/*                       48th edition - Marseille 2025                        */
/*                       Electronique (16) - SyncOrSink                       */
/*                                                                            */
/******************************************************************************/
/* @file SCR/SCR.c                                                            */
/* @authors WorldSkills France "Electronique" skill team                      */
/* @version 1.0                                                               */
/* @date 2025-09-06                                                           */
/*                                                                            */
/* @brief This file contains functions to select and display information      */
/*        pages on the SCR_USR user screen.                                   */
/******************************************************************************/

/* *************************** STANDARD INCLUDES **************************** */

#include <stdio.h>
#include <string.h>

/* **************************** CUSTOM INCLUDES ***************************** */

#include "SCR.h"
#include "ssd1306.h"

/* ******************************** DEFINES ********************************* */

#define C_SCR_NB_HZ_IN_KHZ (1000.0)

/* ========================================================================== */
/* @req SYS_REQ-0401-001 : Paramètres de l’écran utilisateur                  */
/* The SCR_USR user screen shall be able to display 3 lines of 18 characters. */
/* ========================================================================== */
#define C_SCR_USR_FONT     (Font_7x10)
#define C_SCR_USR_NB_CLMNS (18) /* SSD1306_WIDTH = 128px -> 128/font_7px = 18 */
/* ========================================================================== */

#define C_SCR_USR_CALENDAR_NAME_MAX_LENGTH (12)

/* *************************** TYPES DECLARATION **************************** */

/* ========================================================================== */
/* @req SYS_REQ-0401-001 : Paramètres de l’écran utilisateur                  */
/* The SCR_USR user screen shall be able to display 3 lines of 18 characters. */
/* ========================================================================== */
typedef enum
{
    E_SCR_USR_LINE_1 = 0,
    E_SCR_USR_LINE_2,
    E_SCR_USR_LINE_3,
    E_SCR_USR_NB_LINES /* SSD1306_HEIGHT = 32px -> 32/font_10px = 3 */
} tScrUsrLines;
typedef char tScrUsrBufferLine[C_SCR_USR_NB_CLMNS + 1];
typedef tScrUsrBufferLine tScrUsrBuffer[E_SCR_USR_NB_LINES];
/* ========================================================================== */

/* ******************************* CONSTANTS ******************************** */

const unsigned int C_SCR_USR_LINE_VERTICAL_OFFSET[E_SCR_USR_NB_LINES] = {0, 11, 22};

/* **************************** GLOBAL VARIABLES **************************** */

/* **************************** STATIC VARIABLES **************************** */

/* ********************** STATIC FUNCTIONS DECLARATION ********************** */

/* ************************* FUNCTIONS DECLARATION ************************** */

/* ********************** STATIC FUNCTIONS DEFINITION *********************** */

/******************************************************************************/
/* @function __SCR_Page_Home                                                  */
/*                                                                            */
/* @brief Generates user screen content for "Home" page.                      */
/* @param [out] pageContent User screen content                               */
/* @req SYS_REQ-0402-001 : Ecran d’accueil (STATE_HOME)                       */
/******************************************************************************/
static void __SCR_Page_Home(tScrUsrBuffer pageContent)
{
    snprintf(pageContent[E_SCR_USR_LINE_1], sizeof(tScrUsrBufferLine), "WorldSkills France");
    snprintf(pageContent[E_SCR_USR_LINE_2], sizeof(tScrUsrBufferLine), "  CNAT 48 - 2025  ");
    snprintf(pageContent[E_SCR_USR_LINE_3], sizeof(tScrUsrBufferLine), "    REGION xxx    ");
}

/******************************************************************************/
/* @function __SCR_Page_Power                                                 */
/*                                                                            */
/* @brief Generates user screen content for "Power" page.                     */
/* @param [in]  ps_p_voltage Positive power source voltage                    */
/* @param [in]  ps_n_voltage Negative power source voltage                    */
/* @param [out] pageContent User screen content                               */
/* @req SYS_REQ-0404-001 : Ecran alimentation (STATE_PWR)                     */
/******************************************************************************/
static void __SCR_Page_Power
(
    const double ps_p_voltage,
    const double ps_n_voltage,
    tScrUsrBuffer pageContent
)
{
}

/******************************************************************************/
/* @function __SCR_Page_Timers                                                */
/*                                                                            */
/* @brief Generates user screen content for "Timers" page.                    */
/* @param [in] tim22_frequency TIM22 timer frequency                          */
/* @param [out] pageContent    User screen content                            */
/* @req SYS_REQ-0406-002 : Ecran timers (STATE_TIM)                           */
/******************************************************************************/
static void __SCR_Page_Timers
(
	const double tim22_frequency,
    tScrUsrBuffer pageContent
)
{
}

/******************************************************************************/
/* @function __SCR_Page_Carrier                                               */
/*                                                                            */
/* @brief Generates user screen content for "RF carrier" page.                */
/* @param [in] rf_carrier_active RF carrier activation status                 */
/* @param [out] pageContent User screen content                               */
/* @req SYS_REQ-0407-001 : Ecran porteuse (STATE_CAR)                         */
/******************************************************************************/
static void __SCR_Page_Carrier
(
    const bool rf_carrier_active,
    tScrUsrBuffer pageContent
)
{
}

/* ************************** FUNCTIONS DEFINITION ************************** */

/******************************************************************************/
/* @function SCR_DisplayContent                                               */
/*                                                                            */
/* @brief Displays informations on SCR_USR user screen.                       */
/* @param [in] system_state      Current system state                         */
/* @param [in] ps_p_voltage      Positive power source voltage                */
/* @param [in] ps_n_voltage      Negative power source voltage                */
/* @param [in] tim22_frequency   TIM22 timer frequency                        */
/* @param [in] rf_carrier_active RF carrier activation status                 */
/* @pre The SCR_USR screen shall have been configured by SCR_Init function.   */
/* @req SYS_REQ-0401-001 : Paramètres de l’écran utilisateur                  */
/******************************************************************************/
void SCR_DisplayContent
(
    const tCONF_SystemStates system_state,
    const double ps_p_voltage,
    const double ps_n_voltage,
	const double tim22_frequency,
    const bool rf_carrier_active
)
{
    tScrUsrBuffer pageContent;
    int i;

    memset(pageContent, 0, sizeof(tScrUsrBuffer));

    /* ---- Building SCR_USR content ---------------------------------------- */
    switch(system_state)
    {
        case E_STATE_HOME:
            __SCR_Page_Home(pageContent);
            break;
        case E_STATE_TIM:
            __SCR_Page_Timers(tim22_frequency, pageContent);
            break;
        case E_STATE_PWR:
            __SCR_Page_Power(ps_p_voltage, ps_n_voltage, pageContent);
            break;
        case E_STATE_CAR:
            __SCR_Page_Carrier(rf_carrier_active, pageContent);
            break;
        default:
            __SCR_Page_Home(pageContent);
            break;
    }

    /* ---- Writing SCR_USR content on screen ------------------------------- */
    ssd1306_Fill(Black);
    for(i = 0 ; i < E_SCR_USR_NB_LINES ; ++i)
    {
        ssd1306_SetCursor(0, C_SCR_USR_LINE_VERTICAL_OFFSET[i]);
        ssd1306_WriteString(pageContent[i], C_SCR_USR_FONT, White);
    }
    ssd1306_UpdateScreen();
}

/******************************************************************************/
/* @function SCR_Init                                                         */
/*                                                                            */
/* @brief Initializes SCR_USR user screen.                                    */
/* @req SYS_REQ-0401-001 : Paramètres de l’écran utilisateur                  */
/******************************************************************************/
void SCR_Init(void)
{
    ssd1306_Init();
    ssd1306_Fill(Black);
    ssd1306_UpdateScreen();
}

/******************************************************************************/
/*                                                                            */
/*                             WorldSkills France                             */
/*                       48th edition - Marseille 2025                        */
/*                       Electronique (16) - SyncOrSink                       */
/*                                                                            */
/******************************************************************************/
