const express = require('express');
const domainController = require('../controllers/domainController');
const levelController = require('../controllers/levelController');
const questionController = require('../controllers/questionController');
const answerController = require('../controllers/answerController');

const router = express.Router();

/**
 * @openapi
 * /domains:
 *  get:
 *   tags: 
 *    - Domains
 *   description: Get all domains
 *   responses:
 *    200:
 *     description: A successful response
 */
router.get('/domains', domainController.getDomains);
/**
 * @openapi
 * /domains/{id}:
 *  get:
 *   tags:
 *    - Domains
 *   description: Get one domains
 *   parameters:
 *   - in: path
 *     name: id
 *     description: The domain id
 *   responses:
 *    200:
 *     description: A successful response
 */
router.get('/domains/:id', domainController.getDomain);

/**
 * @openapi
 * /levels:
 *  get:
 *   tags: 
 *    - Levels
 *   description: Get all levels
 *   responses:
 *    200:
 *     description: A successful response
 */
router.get('/levels', levelController.getLevels);
/**
 * @openapi
 * /levels/{id}:
 *  get:
 *   tags:
 *    - Levels
 *   description: Get one levels
 *   parameters:
 *   - in: path
 *     name: id
 *     description: The level id
 *   responses:
 *    200:
 *     description: A successful response
 */
router.get('/levels/:id', levelController.getLevel);

/**
 * @openapi
 * /questions:
 *  get:
 *   tags: 
 *    - Questions
 *   description: Get all questions
 *   responses:
 *    200:
 *     description: A successful response
 */
router.get('/questions', questionController.getQuestions);
/**
 * @openapi
 * /questions/{id}:
 *  get:
 *   tags:
 *    - Questions
 *   description: Get one question
 *   parameters:
 *   - in: path
 *     name: id
 *     description: The question id
 *   responses:
 *    200:
 *     description: A successful response
 */
router.get('/questions/:id', questionController.getQuestion);
/**
 * @openapi
 * /questions/{level_id}/{domain_id}:
 *  get:
 *   tags:
 *    - Questions
 *   description: Get some questions by level and domain
 *   parameters:
 *   - in: path
 *     name: level_id
 *     description: The level id
 *   - in: path
 *     name: domain_id
 *     description: The domain id
 *   responses:
 *    200:
 *     description: A successful response
 */
router.get('/questions/:level_id/:domain_id', questionController.getQuestionsByLevelAndDomainAndActive);
/**
 * @openapi
 * /questions/:
 *  post:
 *   tags:
 *    - Questions
 *   description: Add a question
 *   parameters:
 *   - in: body
 *     name: intitule
 *     description: The question text
 *   - in: body
 *     name: infos
 *     description: The question informations 
 *   - in: body
 *     name: redactor
 *     description: The redactor name
 *   - in: body
 *     name: level_id
 *     description: The question level id
 *   - in: body
 *     name: domain_id
 *     description: The question domain id
 *   responses:
 *    201:
 *     description: A successful response
 */
router.post('/questions', questionController.createQuestion);
/**
 * @openapi
 * /questions/{id}:
 *  put:
 *   tags:
 *    - Questions
 *   description: Update a question
 *   parameters:
 *   - in: path
 *     name: id
 *     description: The question id
 *   - in: body
 *     name: intitule
 *     description: The question text
 *   - in: body
 *     name: infos
 *     description: The question informations 
 *   - in: body
 *     name: redactor
 *     description: The redactor name
 *   - in: body
 *     name: level_id
 *     description: The question level id
 *   - in: body
 *     name: domain_id
 *     description: The question domain id
 *   responses:
 *    200:
 *     description: A successful response
 */
router.put('/questions/:id', questionController.updateQuestion);

/**
 * @openapi
 * /answers:
 *  get:
 *   tags: 
 *    - Answers
 *   description: Get all answers
 *   responses:
 *    200:
 *     description: A successful response
 */
router.get('/answers', answerController.getAnswers);
/**
 * @openapi
 * /answers/{id}:
 *  get:
 *   tags:
 *    - Answers
 *   description: Get one answer
 *   parameters:
 *   - in: path
 *     name: id
 *     description: The answer id
 *   responses:
 *    200:
 *     description: A successful response
 */
router.get('/answers/:id', answerController.getAnswer);
/**
 * @openapi
 * /answers:
 *  post:
 *   tags:
 *   - Answers
 *   description: Create an answer
 *   parameters:
 *   - in: body
 *     name: content
 *     description: The answer content
 *   - in: body
 *     name: is_correct
 *     description: The answer is correct
 *   - in: body
 *     name: question_id
 *     description: The question id
 *   responses:
 *    200:
 *     description: A successful response
 */
router.post('/answers', answerController.createAnswer);
/**
 * @openapi
 * /answers{id}:
 *  put:
 *   tags:
 *   - Answers
 *   description: Update an answer
 *   parameters:
 *   - in: path
 *     name: id
 *     description: The answer id
 *   - in: body
 *     name: content
 *     description: The answer content
 *   - in: body
 *     name: is_correct
 *     description: The answer is correct
 *   - in: body
 *     name: question_id
 *     description: The question id
 *   responses:
 *    200:
 *     description: A successful response
 */
router.put('/answers/:id', answerController.updateAnswer);
/**
 * @openapi
 * /answers{id}:
 *  delete:
 *   tags:
 *   - Answers
 *   description: Delete an answer
 *   parameters:
 *   - in: path
 *     name: id
 *     description: The answer id
 *   responses:
 *    200:
 *     description: A successful response
 */
router.delete('/answers/:id', answerController.deleteAnswer);


module.exports = router;
